package ru.yandex.chemodan.xiva;

import java.io.IOException;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.util.http.HttpExceptionResponseHandler;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.io.InputStreamSourceUtils;
import ru.yandex.misc.io.InputStreamXUtils;
import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class BasicXivaClient {
    private static final Logger logger = LoggerFactory.getLogger(BasicXivaClient.class);

    private static final HttpExceptionResponseHandler<XivaSendResponse> BASIC_SEND_HANDLER =
            HttpExceptionResponseHandler.build(BasicXivaClient::parseSendResponse)
                    .build();

    private static final HttpExceptionResponseHandler<XivaSendResponse> BATCH_SEND_HANDLER =
            HttpExceptionResponseHandler.build(BasicXivaClient::parseBatchSendResponse)
                    .build();

    private static final String V2_SEND_PATH = "/v2/send";

    private static final String V2_BATCH_SEND_PATH = "/v2/batch_send";

    private static final String BROADCAST_SEND_PATH = "/beta/wild/send";

    private final String host;

    private final HttpClient httpClient;

    public BasicXivaClient(String host, HttpClient httpClient) {
        this.host = host;
        this.httpClient = httpClient;
    }

    /**
     * Returns Xiva Transit ID for logging
     */
    public XivaSendResponse send(XivaEvent event, String token) {
        byte[] eventBodyBytes = event.serializeBody();
        logger.info("Sending push event={} to recipient={}", event.eventType, event.recipient);
        if (logger.isTraceEnabled()) {
            logger.trace("Push body: {}", new String(eventBodyBytes));
        }

        HttpPost httpPost = new HttpPost(buildV2Url(event));
        httpPost.addHeader(consAuthHeader(token));
        httpPost.setEntity(new ByteArrayEntity(eventBodyBytes, ContentType.APPLICATION_JSON));
        try {
            return httpClient.execute(httpPost, getSendHandler(event));
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public ListF<XivaSubscription> list(PassportUid uid, String token, String service) {
        HttpGet httpGet = new HttpGet(
                consUriBuilder("/v2/list")
                        .addParam("uid", uid)
                        .addParam("service", service)
                        .toUrl()
        );
        httpGet.addHeader(consAuthHeader(token));
        try {
            return httpClient.execute(httpGet, response -> XivaSubscription.parseList(
                    InputStreamSourceUtils.wrap(response.getEntity().getContent())
            ));
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public XivaSecretSign getXivaSecretSign(ListF<String> services, ListF<String> users, String token,
                                            ObjectMapper objectMapper)
    {
        HttpGet httpGet = new HttpGet(
                consUriBuilder("/v2/secret_sign")
                        .addParam("service", services.mkString(","))
                        .addParam("user", users.mkString(","))
                        .toUrl()
        );
        httpGet.addHeader(consAuthHeader(token));
        try {
            return httpClient.execute(httpGet,
                    response -> objectMapper.readValue(response.getEntity().getContent(), XivaSecretSign.class)
            );
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private String buildV2Url(XivaEvent event) {
        UriBuilder builder = consUriBuilder(getSendPath(event.recipient))
                .addParamO("uid", event.recipient.getUrlValueO())
                .addParam("event", event.eventType);

        if (event.originalTimestamp.isPresent()) {
            builder.addParam("event_ts", event.originalTimestamp.get());
        }

        if (event.ttl.isPresent()) {
            builder.addParam("ttl", event.ttl.get());
        }

        return builder.toUrl();
    }

    private UriBuilder consUriBuilder(String path) {
        return UriBuilder.cons(host)
                .appendPath(path);
    }

    private static String getSendPath(XivaEventRecipient recipient) {
        if (recipient.isBroadcast()) {
            return BROADCAST_SEND_PATH;
        } else if (recipient.isBulk()) {
            return V2_BATCH_SEND_PATH;
        } else {
            return V2_SEND_PATH;
        }
    }

    private static HttpExceptionResponseHandler<XivaSendResponse> getSendHandler(XivaEvent event) {
        return event.isBatch() ? BATCH_SEND_HANDLER : BASIC_SEND_HANDLER;
    }

    private static XivaSendResponse parseBatchSendResponse(HttpResponse response) {
        return new XivaSendResponse(consResponseHeaders(response), parseBatchSendResults(response));
    }

    private static ListF<ListF<XivaSendResult>> parseBatchSendResults(HttpResponse response) {
        try {
            return XivaSendResult.parseBatchResponse(
                    InputStreamSourceUtils.wrap(response.getEntity().getContent())
            );
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static XivaSendResponse parseSendResponse(HttpResponse response) {
        return new XivaSendResponse(
                consResponseHeaders(response),
                Cf.<ListF<XivaSendResult>>list(Cf.list(consSendResult(response)))
        );
    }

    private static XivaSendResult consSendResult(HttpResponse response) {
        try {
            return new XivaSendResult(
                    response.getStatusLine().getStatusCode(),
                    InputStreamXUtils.wrap(response.getEntity().getContent())
                            .readString()
            );
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static XivaResponseHeaders consResponseHeaders(HttpResponse resp) {
        return XivaResponseHeaders.cons(resp.getAllHeaders());
    }

    private static Header consAuthHeader(String token) {
        return new BasicHeader(HttpHeaderNames.AUTHORIZATION, "Xiva " + token);
    }
}
