package ru.yandex.chemodan.xiva;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collection;

import org.joda.time.Hours;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.bender.serialize.EmptyMarshallerContext;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class XivaEvent {
    final XivaEventRecipient recipient;
    final String eventType;
    final ListF<String> tags;
    final Option<Instant> originalTimestamp;
    final Option<Hours> ttl;
    final Option<XivaPushBody> body;
    final Option<XivaSubscriptionFilter> subscriptions;

    public XivaEvent(XivaEventRecipient recipient, String eventType) {
        this(recipient, eventType, Option.empty(), Cf.list(), Option.empty(), Option.empty(), Option.empty());
    }

    public XivaEvent(XivaEventRecipient recipient, String eventType, ListF<String> tags) {
        this(recipient, eventType, Option.empty(), tags, Option.empty(), Option.empty(), Option.empty());
    }

    private XivaEvent(XivaEventRecipient recipient, String eventType, Option<XivaPushBody> body, ListF<String> tags,
                      Option<Instant> originalTimestamp, Option<Hours> ttl,
            Option<XivaSubscriptionFilter> subscriptions)
    {
        this.recipient = recipient;
        this.eventType = eventType;
        this.body = body;
        this.tags = tags;
        this.originalTimestamp = originalTimestamp;
        this.ttl = ttl;
        this.subscriptions = subscriptions;
    }

    public XivaEvent withBody(XivaPushBody body) {
        return builder()
                .withBody(body)
                .build();
    }

    public XivaEvent withTags(ListF<String> tags) {
        return builder()
                .withTags(tags)
                .build();
    }

    public XivaEvent addTag(String tag) {
        return builder()
                .addTag(tag)
                .build();
    }

    public XivaEvent addTags(Collection<String> tags) {
        return builder()
                .addTags(tags)
                .build();
    }

    public XivaEvent withOriginalTimestamp(Instant timestamp) {
        return builder()
                .withOriginalTimestamp(timestamp)
                .build();
    }

    public XivaEvent withTtl(Hours ttl) {
        return builder()
                .withTtl(ttl)
                .build();
    }

    public XivaEvent withSubscriptions(XivaSubscriptionFilter filter) {
        return builder()
                .withSubscriptions(filter)
                .build();
    }

    public byte[] serializeBody() {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
            BenderJsonWriter json = BenderJsonWriter.json(baos);
            json.writeObjectStart();
            {
                serializeRecipientsIfNotEmpty(json);
                body.ifPresent(b -> buildPayload(json, b));
                buildTags(json);
                serializeSubscriptionsIfNotEmpty(json);
            }
            json.writeObjectEnd();
            json.flush();
            return baos.toByteArray();
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private void serializeSubscriptionsIfNotEmpty(BenderJsonWriter json) {
        subscriptions.ifPresent(subs -> {
            json.writeFieldName("subscriptions");
            subs.serialize(json);
        });
    }

    private void serializeRecipientsIfNotEmpty(BenderJsonWriter json) {
        ListF<XivaEventRecipient.Single> recipients = recipient.getRecipients();
        if (recipients.isEmpty()) {
            return;
        }

        json.writeFieldName("recipients");
        json.writeArrayStart();
        recipients.forEach(single -> single.writeJsonTo(json));
        json.writeArrayEnd();
    }

    private void buildPayload(BenderJsonWriter json, XivaPushBody pushBody) {
        XivaPushBodyRepack repack = pushBody.getRepack();
        XivaPayloadJsonWriterHelper helper = new XivaPayloadJsonWriterHelper(json,
                // all non-standard and non-special fields MUST be defined in payload,
                // so we include them in payload with name prefixed by service,
                // and then repack here by referencing corresponding field at payload
                () -> repack.writeExtraPayloadFields(json)
        );
        json.writeFieldName("payload");
        pushBody.writePushBodyAsJson(helper.getJsonWriterProxy(), new EmptyMarshallerContext());
        repack.writeRepack(json, helper.getTopLevelFields());
    }

    private void buildTags(BenderJsonWriter json) {
        if (!tags.isEmpty()) {
            json.writeFieldName("tags");
            json.writeArrayStart();
            tags.map(XivaUtils::formatTag).forEach(json::writeString);
            json.writeArrayEnd();
        }
    }

    private Builder builder() {
        return new Builder(this);
    }

    boolean isBroadcast() {
        return recipient.isBroadcast();
    }

    boolean isBatch() {
        return recipient.isBulk();
    }

    public Option<XivaPushBody> getBody() {
        return body;
    }

    public ListF<String> getTags() {
        return tags;
    }

    public static class Builder extends DefaultObject {
        XivaEventRecipient recipient;
        String event;
        Option<XivaPushBody> body = Option.empty();
        ListF<String> tags = Cf.arrayList();
        Option<Instant> originalTimestamp = Option.empty();
        Option<Hours> ttl = Option.empty();
        Option<XivaSubscriptionFilter> subscriptions = Option.empty();

        public Builder() {
        }

        Builder(XivaEventRecipient recipient, String event) {
            this.recipient = recipient;
            this.event = event;
        }

        Builder(XivaEvent event) {
            this.recipient = event.recipient;
            this.event = event.eventType;
            this.body = event.body;
            this.tags = event.tags;
            this.originalTimestamp = event.originalTimestamp;
            this.ttl = event.ttl;
            this.subscriptions = event.subscriptions;
        }

        public Builder withBody(XivaPushBody body) {
            this.body = Option.of(body);
            return this;
        }

        public Builder withTags(ListF<String> tags) {
            this.tags = Cf.toArrayList(tags);
            return this;
        }
        public Builder addTag(String tag) {
            this.tags = this.tags.plus1(tag);
            return this;
        }

        public Builder addTags(Collection<String> tags) {
            this.tags = this.tags.plus(tags);
            return this;
        }

        public Builder withOriginalTimestamp(Instant timestamp) {
            this.originalTimestamp = Option.of(timestamp);
            return this;
        }

        public Builder withTtl(Hours ttl) {
            this.ttl = Option.of(ttl);
            return this;
        }

        public Builder withRecipient(XivaEventRecipient uid) {
            this.recipient = uid;
            return this;
        }

        public Builder withEvent(String event) {
            this.event = event;
            return this;
        }

        public Builder withSubscriptions(XivaSubscriptionFilter subscriptions) {
            this.subscriptions = Option.of(subscriptions);
            return this;
        }

        public XivaEvent build() {
            Validate.notNull(recipient, "uid can't be null");
            Validate.notNull(event, "Event can't be null");
            return new XivaEvent(recipient, event, body, tags, originalTimestamp, ttl, subscriptions);
        }
    }
}
