package ru.yandex.chemodan.xiva;

import java.util.function.Consumer;
import java.util.function.Function;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.lang.DefaultObject;

/**
 * https://push.yandex-team.ru/doc/guide.html#api-reference-repack
 *
 * @author Dmitriy Amelin (lemeh)
 */
public class XivaPushBodyRepack extends DefaultObject {
    private static final String XIVA_PUSH_TOKEN = "::xiva::push_token";

    private static final String XIVA_TRANSIT_ID = "::xiva::transit_id";

    private final MapF<XivaPushService, ServiceRepack> serviceRepacks = Cf.linkedHashMap();

    public void writeExtraPayloadFields(BenderJsonWriter json) {
        serviceRepacks.values()
                .forEach(repack -> repack.writePrefixedCustomFieldsToPayload(json));
    }

    public void writeRepack(BenderJsonWriter json, ListF<String> payloadFields) {
        if (serviceRepacks.isEmpty()) {
            return;
        }

        json.writeFieldName("repack");
        json.writeObjectStart();
        getFixedServiceRepacks().forEach((service, repack) -> repack.writeRepack(json, payloadFields));
        json.writeObjectEnd();
    }

    MapF<XivaPushService, ServiceRepack> getFixedServiceRepacks() {
        if (serviceRepacks.size() == 1 && serviceRepacks.containsKeyTs(XivaPushService.OTHER)) {
            // We need to repack params, but we couldn't add only 'other' part,
            // because 'other' would work only if there are another rules,
            // so we add 'apns' service here like hack for repack.
            // https://st.yandex-team.ru/CHEMODAN-36415
            return serviceRepacks.plus1(XivaPushService.APNS,
                    serviceRepacks.getTs(XivaPushService.OTHER).deepCopy(XivaPushService.APNS)
            );
        } else {
            return serviceRepacks;
        }
    }

    public XivaPushBodyRepack forEachServiceRepack(Consumer<ServiceRepack> consumer) {
        serviceRepacks.values().forEach(consumer);
        return this;
    }

    public XivaPushBodyRepack withServiceRepack(XivaPushService service) {
        serviceRepack(service);
        return this;
    }

    public XivaPushBodyRepack withServiceRepack(XivaPushService service, Consumer<ServiceRepack> consumer) {
        consumer.accept(serviceRepack(service));
        return this;
    }

    public XivaPushBodyRepack withServiceRepackCopiedFromOther(XivaPushService service,
            Consumer<ServiceRepack> consumer)
    {
        consumer.accept(
                serviceRepack(service,
                        srv -> serviceRepacks.getO(XivaPushService.OTHER)
                                .map(serviceRepack -> serviceRepack.deepCopy(srv))
                                .getOrElse(() -> new ServiceRepack(srv))
                )
        );
        return this;
    }

    private ServiceRepack serviceRepack(XivaPushService service) {
        return serviceRepack(service, ServiceRepack::new);
    }

    private ServiceRepack serviceRepack(XivaPushService service, Function<XivaPushService, ServiceRepack> consF) {
        return serviceRepacks.computeIfAbsent(service, consF);
    }

    public XivaPushBodyRepack includeTransitIdAndAllPayloadFieldsToAllServices() {
        return includeSpecialAndAllPayloadFieldsToAllServices(repack ->
                repack.includeTransitId("transit_id"));
    }

    public XivaPushBodyRepack includeSpecialAndAllPayloadFieldsToAllServices() {
        return includeSpecialAndAllPayloadFieldsToAllServices(repack ->
                repack.includePushToken("r").includeTransitId("transit_id"));
    }

    private XivaPushBodyRepack includeSpecialAndAllPayloadFieldsToAllServices(Consumer<ServiceRepack> consumer) {
        serviceRepack(XivaPushService.OTHER);
        return forEachServiceRepack(repack -> consumer.accept(repack.includeAllPayloadFields()));
    }

    public XivaPushBodyRepack includeAllPayloadFieldsToAllServices() {
        return forEachServiceRepack(ServiceRepack::includeAllPayloadFields);
    }

    public XivaPushBodyRepack addCustomFields(MapF<XivaPushService, MapF<String, Object>> customFields) {
        customFields.forEach((service, fields) -> serviceRepack(service).addCustomFields(fields));
        return this;
    }

    public XivaPushBodyRepack withApnsSilent() {
        return withServiceRepackCopiedFromOther(XivaPushService.APNS, serviceRepack -> serviceRepack.addStandardField("aps",
                Cf.map(
                        "apns-push-type", "background",
                        "apns-priority", 5
                )
        ));
    }

    public static class ServiceRepack extends DefaultObject {
        private final XivaPushService service;

        private final MapF<String, Object> standardExtraFields;

        private final MapF<String, Object> customExtraFields;

        private final ListF<String> includedPayloadFields;

        private final MapF<String, String> repackedPayloadFields;

        private boolean includeAllPayloadFields;

        ServiceRepack(XivaPushService service) {
            this(service, Cf.linkedHashMap(), Cf.linkedHashMap(), Cf.arrayList(), Cf.linkedHashMap(), false);
        }

        private ServiceRepack(XivaPushService service,
                MapF<String, Object> standardExtraFields, MapF<String, Object> customExtraFields,
                ListF<String> includedPayloadFields, MapF<String, String> repackedPayloadFields,
                boolean includeAllPayloadFields)
        {
            this.service = service;
            this.standardExtraFields = standardExtraFields;
            this.customExtraFields = customExtraFields;
            this.includedPayloadFields = includedPayloadFields;
            this.repackedPayloadFields = repackedPayloadFields;
            this.includeAllPayloadFields = includeAllPayloadFields;
        }

        private ServiceRepack deepCopy(@SuppressWarnings("SameParameterValue") XivaPushService service) {
            return new ServiceRepack(
                    service,
                    copyAsLinkedMap(standardExtraFields),
                    copyAsLinkedMap(customExtraFields),
                    Cf.toArrayList(includedPayloadFields),
                    copyAsLinkedMap(repackedPayloadFields),
                    includeAllPayloadFields
            );
        }

        public XivaPushService getService() {
            return service;
        }

        public ServiceRepack addStandardField(String field, Object value) {
            standardExtraFields.put(field, value);
            return this;
        }

        public ServiceRepack addCustomFields(MapF<String, Object> fields) {
            customExtraFields.putAll(fields);
            return this;
        }

        public ServiceRepack includePushToken(String newField) {
            return repackPayloadField(newField, XIVA_PUSH_TOKEN);
        }

        public ServiceRepack includeTransitId(String newField) {
            return repackPayloadField(newField, XIVA_TRANSIT_ID);
        }

        public ServiceRepack repackPayloadField(String newField, String srcField) {
            repackedPayloadFields.put(newField, srcField);
            return this;
        }

        public ServiceRepack includePayloadField(String field) {
            includedPayloadFields.add(field);
            return this;
        }

        public ServiceRepack includeAllPayloadFields() {
            includeAllPayloadFields = true;
            return this;
        }

        // all custom (non-standard and non-special) fields MUST be defined in payload,
        // so we include them in payload with name prefixed by service,
        // and then repack to original name in each service
        private void writePrefixedCustomFieldsToPayload(BenderJsonWriter json) {
            customExtraFields.forEach((field, value) -> {
                json.writeFieldName(addServicePrefix(field));
                json.writeString(value.toString());
            });
        }

        private void writeCustomFieldsRepackFromPrefixedNames(BenderJsonWriter json) {
            customExtraFields.keys().forEach(field -> writeKeyValueObject(json, field, addServicePrefix(field)));
        }

        private void writeRepack(BenderJsonWriter json, ListF<String> allPayloadFields) {
            json.writeFieldName(service.value());

            json.writeObjectStart();

            XivaSerializeUtil.writeMapContents(json, standardExtraFields);

            json.writeFieldName("repack_payload");
            json.writeArrayStart();

            (includeAllPayloadFields ? allPayloadFields : includedPayloadFields).forEach(json::writeString);
            repackedPayloadFields.forEach((field, value) -> writeKeyValueObject(json, field, value));
            writeCustomFieldsRepackFromPrefixedNames(json);

            json.writeArrayEnd();
            json.writeObjectEnd();
        }

        private String addServicePrefix(String field) {
            return service + "_" + field;
        }
    }

    private static void writeKeyValueObject(BenderJsonWriter json, String field, String value) {
        json.writeObjectStart();
        json.writeFieldName(field);
        json.writeString(value);
        json.writeObjectEnd();
    }

    private static <K, V> MapF<K, V> copyAsLinkedMap(MapF<K, V> map) {
        MapF<K, V> result = Cf.linkedHashMap();
        result.putAll(map);
        return result;
    }
}
