package ru.yandex.chemodan.zk;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.function.Function;

import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.Interval;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.commune.zk2.ZkException;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.ZkPathUtils;
import ru.yandex.commune.zk2.ZkWatcher;
import ru.yandex.commune.zk2.ZkWrapper;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.TimeoutRuntimeException;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FakeInMemoryZkWrapper implements ZkWrapper {
    private static final Logger logger = LoggerFactory.getLogger(FakeInMemoryZkWrapper.class);

    private static final Duration TIMEOUT = Duration.standardSeconds(2);

    private static final Duration SLEEP_BETWEEN_INIT_CHECKS = new Duration(10);

    private final Node rootNode = new Node(new ZkPath("/"));

    private final Watchers existsWatchers = new Watchers();

    private final Watchers getChildrenWatchers = new Watchers();

    private final Watchers getDataWatchers = new Watchers();

    public void initialize(ZkRegistry registry) {
        registry.connected(this);
        Instant start = new Instant();
        while (!registry.isInitialized()) {
            if (new Interval(start, new Instant()).toDuration().isLongerThan(TIMEOUT)) {
                throw new TimeoutRuntimeException();
            }

            try {
                Thread.sleep(SLEEP_BETWEEN_INIT_CHECKS.getMillis());
            } catch (InterruptedException e) {
                throw ExceptionUtils.translate(e);
            }
        }
    }

    @Override
    public ListF<ZkPath> getChildren(ZkPath path, Option<ZkWatcher> watcher) {
        logActionWithWatcher("getChildren", path, watcher);
        return getNodeData(path, node -> {
            getChildrenWatchers.add(path, watcher);
            return node.getChildrenPaths();
        });
    }

    @Override
    public Tuple2<Option<byte[]>, Stat> getData(ZkPath path, Option<ZkWatcher> watcher) {
        logActionWithWatcher("getData", path, watcher);
        return getNodeData(path, node -> {
            getDataWatchers.add(path, watcher);
            return node.getDataAndStat();
        });
    }

    @Override
    public Stat setData(ZkPath path, Option<byte[]> data, Option<Integer> version) {
        logActionWithVersion("setData", path, version);
        return getNodeData(path, node -> {
            node.setData(data);
            getDataWatchers.fireEvents(path, Watcher.Event.EventType.NodeDataChanged);
            return node.getStat();
        });
    }

    @Override
    public Option<Stat> exists(ZkPath path, Option<ZkWatcher> watcher) {
        logActionWithWatcher("exists", path, watcher);
        existsWatchers.add(path, watcher);
        return getNodeO(path)
                .map(Node::getStat);
    }

    @Override
    public ZkPath create(ZkPath path, Option<byte[]> data, CreateMode mode) {
        logger.info("Create node at path = {} with mode = {}", path, mode);
        if (path.isRoot()) {
            throw consZkException(KeeperException.Code.NODEEXISTS, path);
        }

        return getNodeData(path.parent(), node -> {
            Node child = node.addChild(path.getName(), data);
            getChildrenWatchers.fireEvents(path.parent(), Watcher.Event.EventType.NodeChildrenChanged);
            existsWatchers.fireEvents(path, Watcher.Event.EventType.NodeCreated);
            return child.getPath();
        });
    }

    @Override
    public void delete(ZkPath path, Option<Integer> version) {
        logActionWithVersion("delete", path, version);
        getNodeData(path.parent(), node -> {
            Node child = node.remove(path.getName());
            getChildrenWatchers.fireEvents(path.parent(), Watcher.Event.EventType.NodeChildrenChanged);
            existsWatchers.fireEvents(path, Watcher.Event.EventType.NodeDeleted);
            return child;
        });
    }

    @Override
    public void close() {
        logger.info("Close called");
    }

    private <T> T getNodeData(ZkPath path, Function<Node, T> function) {
        return function.apply(getNode(path));
    }

    private Node getNode(ZkPath path) {
        return getNodeO(path)
                .getOrThrow(() -> consZkException(KeeperException.Code.NONODE, path));
    }

    private Option<Node> getNodeO(ZkPath path) {
        return rootNode.getDescendantOrSelf(ZkPathUtils.pathElements(path).iterator());
    }

    private static ZkException consZkException(KeeperException.Code code, ZkPath path) {
        return new ZkException(KeeperException.create(code, path.getPath()));
    }

    private static void logActionWithWatcher(String action, ZkPath path, Option<ZkWatcher> watcher) {
        logger.info("{} for path = {} with watcher = {}", action, path, toStringOrNone(watcher));
    }

    private static void logActionWithVersion(String action, ZkPath path, Option<Integer> version) {
        logger.info("{} for path = {} with version = {}", action, path, toStringOrNone(version));
    }

    private static String toStringOrNone(Option<?> object) {
        return object.map(Object::toString).getOrElse("none");
    }

    private static class Watchers {
        private final MapF<String, ListF<ZkWatcher>> watchers = Cf.wrap(new ConcurrentHashMap<>());

        void add(ZkPath path, Option<ZkWatcher> watcherO) {
            watcherO.ifPresent(watcher -> add(path, watcher));
        }

        void add(ZkPath path, ZkWatcher watcher) {
            watchers.computeIfAbsent(path.getPath(), s -> Cf.wrap(new CopyOnWriteArrayList<>()))
                    .add(watcher);
        }

        ListF<ZkWatcher> get(ZkPath path) {
            return watchers.getO(path.getPath())
                    .getOrElse(Cf.list());
        }

        void fireEvents(ZkPath path, Watcher.Event.EventType eventType) {
            WatchedEvent watchedEvent = new WatchedEvent(
                    eventType,
                    Watcher.Event.KeeperState.SyncConnected,
                    path.getPath()
            );
            get(path).forEach(watcher -> watcher.process(watchedEvent));
        }
    }

    private static class Node extends DefaultObject {
        final ZkPath path;

        final MapF<String, Node> children = Cf.wrap(new ConcurrentHashMap<>());

        volatile Option<byte[]> data = Option.empty();

        Node(ZkPath path) {
            this.path = path;
        }

        public Node(ZkPath path, Option<byte[]> data) {
            this(path);
            this.data = data;
        }

        Option<Node> getDescendantOrSelf(IteratorF<String> names) {
            return names.hasNext()
                    ? getO(names.next()).filterMap(node -> node.getDescendantOrSelf(names))
                    : Option.of(this);
        }

        Option<Node> getO(String name) {
            return children.getO(name);
        }

        ListF<ZkPath> getChildrenPaths() {
            return getChildren().map(Node::getPath);
        }

        ListF<Node> getChildren() {
            return children.values().toList();
        }

        ZkPath getPath() {
            return path;
        }

        Node addChild(String name, Option<byte[]> data) {
            Node child = new Node(path.child(name), data);
            Object prevValue = children.putIfAbsent(name, child);
            if (prevValue != null) {
                throw consZkException(KeeperException.Code.NODEEXISTS, child.getPath());
            }
            return child;
        }

        public Node remove(String name) {
            Node child = children.removeTs(name);
            if (child == null) {
                throw consZkException(KeeperException.Code.NONODE, path.child(name));
            }
            return child;
        }

        public void setData(Option<byte[]> data) {
            this.data = data;
        }

        public Tuple2<Option<byte[]>, Stat> getDataAndStat() {
            return new Tuple2<>(data, getStat());
        }

        // TODO: impl
        public Stat getStat() {
            return new Stat();
        }
    }
}
