package ru.yandex.chemodan.eventlog;

import org.joda.time.Duration;
import org.joda.time.ReadableDuration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.YandexCloudRequestId;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.chemodan.eventlog.log.TskvEventType;
import ru.yandex.chemodan.log.TskvNdcUtil;
import ru.yandex.chemodan.monica.intervaldistribution.ComparableIntervalMetricMapWithDistribution;
import ru.yandex.chemodan.monica.intervaldistribution.TimeUtils;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public abstract class EventLogListenerSupport implements LogListener, MonicaContainer {
    private static final Logger logger = LoggerFactory.getLogger(EventLogListenerSupport.class);

    static final String UNSPECIFIED_EVENT_TYPE_VALUE = "unspecified";

    @MonicaMetric(description = "Parse errors")
    @GroupByDefault
    final MeterMap parseErrors = new MeterMap();

    @MonicaMetric(description = "Unknown event types")
    @GroupByDefault
    final MeterMap unknownTypes = new MeterMap();

    @MonicaMetric(description = "Skipped events")
    @GroupByDefault
    final MeterMap skippedEvents = new MeterMap();

    @MonicaMetric(description = "Event processing time lag thresholds")
    @GroupByDefault
    final ComparableIntervalMetricMapWithDistribution<ReadableDuration> eventTimeLagThresholds =
            new ComparableIntervalMetricMapWithDistribution<>(
                    Cf.list(
                            Duration.standardSeconds(20),
                            Duration.standardSeconds(40),
                            Duration.standardMinutes(1),
                            Duration.standardMinutes(3),
                            Duration.standardMinutes(7),
                            Duration.standardMinutes(15),
                            Duration.standardMinutes(30)
                    ),
                    duration -> TimeUtils.formatSimple(duration.toPeriod())
            );

    @MonicaMetric(description = "Event processing time lag statistic")
    @GroupByDefault
    final Statistic eventTimeLag = new Statistic();

    @MonicaMetric(description = "Event processing time lead")
    @GroupByDefault
    final Statistic eventTimeLead = new Statistic();

    @MonicaMetric(description = "Event processing")
    @GroupByDefault
    final InstrumentMap eventProcessing = new InstrumentMap();

    public abstract void processEvent(AbstractEvent event);

    protected Logger loggerForMonitoring() {
        return logger;
    }

    @Override
    public final void processLogLine(String line) {
        TskvEventLogLine logLine = TskvEventLogLine.parse(line);
        String eventTypeStr = logLine.getEventTypeStrO().getOrElse(UNSPECIFIED_EVENT_TYPE_VALUE);

        Option<AbstractEvent> eventO;
        try {
            eventO = logLine.toEvent();
        } catch(RuntimeException e) {
            parseErrors.inc(eventTypeStr);
            parseErrors.inc();
            loggerForMonitoring().error("Error while parsing event of type={}, {}: {}", eventTypeStr, line, e);
            return;
        }

        if (!eventO.isPresent()) {
            unknownTypes.inc(eventTypeStr);
            unknownTypes.inc();
            loggerForMonitoring().warn("Unknown event type={}, {}", eventTypeStr, line);
            return;
        }

        if (shouldSkipCompletely(eventO.get())) {
            return;
        }

        Duration receiveDelay = eventO.get().metadata.getReceiveDelay();
        long receiveDelayMillis = receiveDelay.getMillis();
        eventTimeLagThresholds.inc(receiveDelay);

        if (receiveDelayMillis >= 0) {
            eventTimeLag.update(receiveDelayMillis);
            eventTimeLead.update(0);
        } else {
            // normally time lag must be >> 0 - this could occur due to problems with real time clock sync
            eventTimeLag.update(0);
            eventTimeLead.update(Math.abs(receiveDelayMillis));
        }

        withEventNdc(eventO.get(), () -> {
            loggerForMonitoring().info("Event time lag: {} s", receiveDelay.getStandardSeconds());

            eventProcessing.measure(
                    () -> processEvent(eventO.get()),
                    new MetricName(eventO.get().getEventType().toString().toLowerCase()),
                    UpdateMode.RECURSIVE);
        });
    }

    protected void withEventNdc(AbstractEvent event, Runnable runnable) {
        Tuple2List<String, Object> values = getNdcValues(event);

        if (values.isNotEmpty()) {
            TskvNdcUtil.runWithNdc(runnable, getNdcValues(event));
        } else {
            runnable.run();
        }
    }

    protected Tuple2List<String, Object> getNdcValues(AbstractEvent event) {
        return Tuple2List.fromPairs(
                "history_lag", event.metadata.getReceiveDelay().getStandardSeconds(),
                "history_hostname", event.metadata.host.getOrElse(""),
                "history_ycrid", event.metadata.cloudRequestId.map(YandexCloudRequestId::toString).getOrElse("")
        );
    }

    protected void incSkippedEvents(TskvEventType eventType) {
        incSkippedEvents(eventType, 1);
    }

    protected void incSkippedEvents(TskvEventType eventType, int eventCount) {
        if (eventCount <= 0) {
            return;
        }
        skippedEvents.inc(eventCount, eventType.value());
        skippedEvents.inc(eventCount);
    }

    protected boolean shouldSkipCompletely(AbstractEvent event) {
        return false;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "event-loader",
                new MetricName("event-loader", "log-processing"),
                "Event history log processing");
    }
}
