package ru.yandex.chemodan.eventlog;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.client.MetricsEvent;
import ru.yandex.chemodan.eventlog.events.client.MetricsEventType;
import ru.yandex.chemodan.eventlog.log.TskvClientMetricsEventLogLine;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaContainer;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.MeterMap;
import ru.yandex.misc.monica.core.blocks.UpdateMode;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

public abstract class MetricsEventLogListener implements LogListener, MonicaContainer {

    private static final Logger logger = LoggerFactory.getLogger(MetricsEventLogListener.class);

    static final String UNSPECIFIED_EVENT_TYPE_VALUE = "unspecified";

    @MonicaMetric(description = "Parse errors")
    @GroupByDefault
    final MeterMap parseErrors = new MeterMap();

    @MonicaMetric(description = "Unknown event types")
    @GroupByDefault
    final MeterMap unknownTypes = new MeterMap();

    @MonicaMetric(description = "Skipped events")
    @GroupByDefault
    final MeterMap skippedEvents = new MeterMap();

    @MonicaMetric(description = "Event processing")
    @GroupByDefault
    final InstrumentMap eventProcessing = new InstrumentMap();

    public abstract void processEvent(MetricsEvent event);

    protected Logger loggerForMonitoring() {
        return logger;
    }

    protected boolean filterLogLine(TskvClientMetricsEventLogLine logLine) {
        if (!logLine.getEventTypeStrO().filter("EVENT_CLIENT"::equals).isPresent()) {
            return false;
        }
        if (!logLine.getNonEmptyStringO(MetricsEvent.API_KEY_FIELD).filter("18895"::equals).isPresent()) {
            return false;
        }
        if (!logLine.getNonEmptyStringO(MetricsEvent.ACCOUNT_ID_FIELD).flatMapO(Cf.Long::parseSafe).isPresent()) {
            return false;
        }
        return true;
    }

    @Override
    public final void processLogLine(String line) {
        TskvClientMetricsEventLogLine logLine = TskvClientMetricsEventLogLine.parse(line);
        String eventTypeStr = logLine.getEventTypeStrO().getOrElse(UNSPECIFIED_EVENT_TYPE_VALUE);
        if (!filterLogLine(logLine)) {
            incSkippedEvents(MetricsEventType.UNKNOWN);
            return;
        }
        Option<MetricsEvent> eventO;
        try {
            eventO = logLine.toMetricsEvent();
        } catch(RuntimeException e) {
            parseErrors.inc(eventTypeStr);
            parseErrors.inc();
            loggerForMonitoring().error("Error while parsing metrics event of type={}, {}: {}", eventTypeStr, line, e);
            return;
        }

        if (!eventO.isPresent()) {
            unknownTypes.inc(eventTypeStr);
            unknownTypes.inc();
            return;
        }
        MetricsEvent event = eventO.get();

        if (shouldSkipCompletely(event)) {
            return;
        }
        eventProcessing.measure(
                () -> processEvent(event),
                new MetricName(event.getType().value().toLowerCase()),
                UpdateMode.RECURSIVE);
    }


    protected void incSkippedEvents(MetricsEventType eventType) {
        incSkippedEvents(eventType, 1);
    }

    protected void incSkippedEvents(MetricsEventType eventType, int eventCount) {
        if (eventCount <= 0) {
            return;
        }
        skippedEvents.inc(eventCount, eventType.value());
        skippedEvents.inc(eventCount);
    }

    protected boolean shouldSkipCompletely(MetricsEvent event) {
        return false;
    }

    @Override
    public MetricGroupName groupName(String instanceName) {
        return new MetricGroupName(
                "metrics-event-loader",
                new MetricName("metrics-event-loader", "log-processing"),
                "Metrics event history log processing");
    }
}
