package ru.yandex.chemodan.eventlog.celery;

import java.util.function.Function;

import ru.yandex.chemodan.log.LoggerProxies;
import ru.yandex.chemodan.queller.worker.CeleryOnetimeTask;
import ru.yandex.chemodan.queller.worker.CeleryTaskManager;
import ru.yandex.chemodan.util.AppNameHolder;
import ru.yandex.commune.bazinga.impl.FullJobId;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.commune.dynproperties.DynamicPropertyStaticRegistry;
import ru.yandex.commune.salr.logreader.LogListener;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author dbrylev
 */
public class QuellerFallbackEventsLogListener implements LogListener {
    private static final Logger logger = LoggerFactory.getLogger(QuellerFallbackEventsLogListener.class);
    private static final Logger retryLogger = LoggerProxies.noStackTrace(CeleryTaskManager.class);

    public static final DynamicProperty<Boolean> logLines =
            new DynamicProperty<>(AppNameHolder.get() + "-log-lines", false);

    public static final DynamicProperty<Boolean> allToQueue =
            new DynamicProperty<>(AppNameHolder.get() + "-all-lines-to-queue", false);

    static {
        DynamicPropertyStaticRegistry.addStaticFields(QuellerFallbackEventsLogListener.class);
    }

    private final LogListener listener;
    private final CeleryTaskManager celeryTaskManager;
    private final int retryCount;
    private final Function<String, CeleryOnetimeTask> logLineToCeleryTaskF;

    public QuellerFallbackEventsLogListener(
            LogListener listener,
            CeleryTaskManager celeryTaskManager,
            int quellerRetryCount,
            Function<String, CeleryOnetimeTask> logLineToCeleryTaskF)
    {
        this.listener = listener;
        this.celeryTaskManager = celeryTaskManager;
        this.retryCount = quellerRetryCount;
        this.logLineToCeleryTaskF = logLineToCeleryTaskF;
    }

    @Override
    public void processLogLine(String line) {
        if (allToQueue.get()) {
            sendToQueue(line);
        } else {
            doProcessLogLine(line);
        }
    }

    private FullJobId sendToQueue(String line) {
        return RetryUtils.retryOrThrow(retryLogger, retryCount,
                () -> celeryTaskManager.submit(logLineToCeleryTaskF.apply(line))
        );
    }

    private void doProcessLogLine(String line) {
        try {
            listener.processLogLine(line);

            if (logLines.get()) {
                logger.info("Processed line: {}", line);
            }
        } catch (Throwable t) {
            ExceptionUtils.throwIfUnrecoverable(t);
            FullJobId id = sendToQueue(line);
            logger.info("Failed line processing submitted into queller {}: {}", id, ExceptionUtils.getAllMessages(t));
        }
    }
}
