package ru.yandex.chemodan.eventlog.events;

import org.joda.time.DateTimeConstants;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.joda.time.ReadableDuration;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.log.TskvEventType;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public class EventMetadata extends DefaultObject {
    static final DynamicProperty<Integer> versionAdvance = DynamicProperty.cons("eventHistoryVersionAdvance", 10);

    @BenderIgnore
    public final transient TskvEventType tskvEventType;

    @BenderIgnore
    public final MpfsUid uid;

    @BenderPart(name = "event_timestamp", strictName = true)
    public final Instant commitTime;

    private final transient Instant receiveTime;

    /**
     * Used for making request snapshots. Only events with version < request_time hit request.
     */
    @BenderPart(name = "version", strictName = true)
    public final transient Instant version;

    public final Option<YandexCloudRequestId> cloudRequestId;

    public final Option<String> host;

    public EventMetadata(MpfsUid uid, Instant commitTime, YandexCloudRequestId cloudRequestId) {
        this(uid, commitTime, Option.of(cloudRequestId));
    }

    public EventMetadata(MpfsUid uid, Instant commitTime, Option<YandexCloudRequestId> cloudRequestId) {
        this(TskvEventType.STUB, uid, commitTime, Option.empty(), Option.empty(), cloudRequestId, Option.empty());
    }

    public EventMetadata(
            TskvEventType tskvEventType, MpfsUid uid, Instant commitTime, Option<YandexCloudRequestId> cloudRequestId)
    {
        this(tskvEventType, uid, commitTime, Option.empty(), Option.empty(), cloudRequestId, Option.empty());
    }

    EventMetadata(TskvEventType tskvEventType, MpfsUid uid, Instant commitTime, Instant receiveTime, Instant version,
            Option<YandexCloudRequestId> cloudRequestId, Option<String> host)
    {
        this(tskvEventType, uid, commitTime, Option.of(receiveTime), Option.of(version), cloudRequestId, host);
    }

    EventMetadata(TskvEventType type, MpfsUid uid, Instant commitTime,
            Option<Instant> receiveTime, Option<Instant> version,
            Option<YandexCloudRequestId> cloudRequestId, Option<String> host)
    {
        this.tskvEventType = type;
        this.uid = uid;
        this.commitTime = commitTime;
        this.receiveTime = receiveTime.getOrElse(Instant::new);
        this.version = version.getOrElse(getVersion(this.receiveTime));
        this.cloudRequestId = cloudRequestId;
        this.host = host;
    }

    public EventMetadata withUid(MpfsUid uid) {
        return new EventMetadata(tskvEventType, uid, commitTime, receiveTime, version, cloudRequestId, host);
    }

    public EventMetadata withHost(Option<String> host) {
        return new EventMetadata(tskvEventType, uid, commitTime, receiveTime, version, cloudRequestId, host);
    }

    @SuppressWarnings("unused")
    @BenderPart(name = "platform", strictName = true)
    public Option<String> getPlatform() {
        return cloudRequestId.map(requestId -> requestId.platform);
    }

    public long getTimestampInSeconds() {
        return commitTime.getMillis() / DateTimeConstants.MILLIS_PER_SECOND;
    }

    public Duration getReceiveDelay() {
        return new Duration(commitTime, receiveTime);
    }

    private static Instant getVersion(Instant receiveTime) {
        // intentionally using time in future, so events would not magically appear in request after indexing
        return receiveTime.plus(getVersionAdvance());
    }

    private static Duration getVersionAdvance() {
        return Duration.standardSeconds(versionAdvance.get());
    }

    public EventMetadata subtractFromCommitTime(ReadableDuration duration) {
        return new EventMetadata(
                tskvEventType,
                uid,
                commitTime.minus(duration),
                Option.of(receiveTime),
                Option.of(version),
                cloudRequestId,
                host
        );
    }
}
