package ru.yandex.chemodan.eventlog.events;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.enums.EnumUtils;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.enums.StringEnumResolver;

/**
 * @author Dmitriy Amelin (lemeh)
 * also @see ru.yandex.chemodan.eventlog.log.TskvEventType
 */
public enum EventType implements StringEnum {
    // События файловой системы
    FS_MKDIR(EventCategory.FS), // создать директорию
    FS_MKSYSDIR(EventCategory.FS, Param.REJECT), // создать системную директорию
    FS_COPY(EventCategory.FS), // скопировать ресурс
    FS_MOVE(EventCategory.FS), // переместить ресурс
    FS_RENAME(EventCategory.FS), // переименовать ресурс = переместить, где source.parent = target.parent
    FS_RM(EventCategory.FS), // удалить ресурс
    FS_STORE(EventCategory.FS), // загрузить ресурс (subtype='disk', type='store')
    FS_STORE_UPDATE(EventCategory.FS), // обновить ресурс
    FS_STORE_DOWNLOAD(EventCategory.FS), // сохранить на диск
    FS_STORE_PHOTOSTREAM(EventCategory.FS), // автосохранение изображений и видео
    FS_STORE_PHOTOUNLIM(EventCategory.FS), // автосохранение в специальную unlim_area
    FS_STORE_OFFICE(EventCategory.FS, Param.REJECT), // редактирование офисных файлов
    FS_TRASH_APPEND(EventCategory.FS), // переместить в корзину
    FS_TRASH_RESTORE(EventCategory.FS), // восстановить ресурс из корзины
    FS_TRASH_DROP(EventCategory.FS), // удалить ресурс из корзины
    FS_TRASH_DROP_ALL(EventCategory.FS), // очистить корзину

    // События публикации
    FS_SET_PUBLIC(EventCategory.FS), // опубликовать ресурс
    FS_SET_PRIVATE(EventCategory.FS), // отменить публикацию

    FS_AVIARY_RENDER(EventCategory.FS), // редактирование в Aviary

    // Импорт и экспорт в соц. сети
    SOCIAL_IMPORT(EventCategory.SOCIAL),
    SOCIAL_EXPORT(EventCategory.SOCIAL),

    // События общих папок
    SHARE_CREATE_GROUP(EventCategory.SHARE, Param.REJECT), // сделать папку общей
    SHARE_UNSHARE_FOLDER_USER(EventCategory.SHARE), // отменить общий доступ к папке для всех с указанием пользователя
    SHARE_INVITE_USER(EventCategory.SHARE), // пригласить пользователя к управлению общей папкой
    SHARE_REMOVE_INVITE(EventCategory.SHARE), // отменить приглашение пользователя к управлению общей папкой
    SHARE_CHANGE_INVITE_RIGHTS(EventCategory.SHARE), // изменить права доступа к общей папке для инвайта
    SHARE_CHANGE_RIGHTS(EventCategory.SHARE), // изменить права доступа к общей папке
    SHARE_KICK_FROM_GROUP(EventCategory.SHARE), // отменить доступ пользователя к папке
    SHARE_CHANGE_GROUP_OWNER(EventCategory.SHARE, Param.REJECT), // изменить владельца общей папки
    SHARE_ACTIVATE_INVITE(EventCategory.SHARE), // принять приглашение к общей папке
    SHARE_REJECT_INVITE(EventCategory.SHARE), // отклонить приглашение к общей папке
    SHARE_LEAVE_GROUP(EventCategory.SHARE), // отказаться от доступа к общей папке

    // События альбомов
    ALBUM_CREATE(EventCategory.ALBUM), // создать пустой альбом
    ALBUM_REMOVE(EventCategory.ALBUM), // удалить альбом
    ALBUM_ITEMS_APPEND(EventCategory.ALBUM), // добавить медиа-файл в альбом
    ALBUM_ITEMS_REMOVE(EventCategory.ALBUM), // удалить медиа-файл из альбома
    ALBUM_CHANGE_PUBLICITY(EventCategory.ALBUM), // опубликовать альбом / отменить публикацию альбома
    ALBUM_CHANGE_COVER(EventCategory.ALBUM), // изменить обложку
    ALBUM_CHANGE_COVER_OFFSET(EventCategory.ALBUM, Param.REJECT), // изменить смещение обложки
    ALBUM_CHANGE_TITLE(EventCategory.ALBUM), // изменить заголовок альбома
    ALBUM_POST_TO_SOCIAL(EventCategory.ALBUM), // опубликовать альбом

    // События биллинга
    BILLING_ORDER_NEW(EventCategory.BILLING, Param.REJECT),
    BILLING_BUY_NEW(EventCategory.BILLING),
    BILLING_PROLONG(EventCategory.BILLING),
    BILLING_PROLONG_ERROR(EventCategory.BILLING),
    BILLING_SUBSCRIBE(EventCategory.BILLING),
    BILLING_UNSUBSCRIBE(EventCategory.BILLING),
    BILLING_DELETE(EventCategory.BILLING, Param.REJECT),

    // Инвайты
    INVITE_SENT(EventCategory.INVITE),
    INVITE_ACTIVATION(EventCategory.INVITE),

    // Дисковое пространство
    SPACE_PROMO_ENLARGE(EventCategory.SPACE),
    SPACE_PROMO_REDUCE(EventCategory.SPACE),
    SPACE_PRODUCT_REDUCE(EventCategory.SPACE, Param.REJECT),

    // Комментарии
    COMMENT_ADD(EventCategory.COMMENT),
    COMMENT_DELETE(EventCategory.COMMENT),
    COMMENT_RESTORE(EventCategory.COMMENT),
    COMMENT_LIKE_ADD(EventCategory.COMMENT),
    COMMENT_LIKE_DELETE(EventCategory.COMMENT),
    COMMENT_DISLIKE_ADD(EventCategory.COMMENT),
    COMMENT_DISLIKE_DELETE(EventCategory.COMMENT),

    // Прочие ивенты
    PUBLIC_VISIT(EventCategory.SKIP, Param.REJECT),
    DELETE_SUBDIR(EventCategory.SKIP, Param.REJECT),

    // Специальный тип, сигнализирующий, что событие не нужно индексировать
    SKIP(EventCategory.SKIP, Param.REJECT)
    ;

    private static final SetF<EventType> GROUP_TYPES = Cf.set(
            FS_MKDIR, FS_COPY, FS_MOVE, FS_RM, FS_TRASH_APPEND, FS_TRASH_RESTORE, FS_TRASH_DROP,
            FS_STORE, FS_STORE_UPDATE, FS_STORE_DOWNLOAD, FS_STORE_PHOTOSTREAM, FS_STORE_OFFICE,
            SOCIAL_IMPORT, SOCIAL_EXPORT,
            ALBUM_CREATE, ALBUM_ITEMS_APPEND, ALBUM_ITEMS_REMOVE,
            SHARE_UNSHARE_FOLDER_USER,
            COMMENT_ADD, COMMENT_DELETE, COMMENT_RESTORE,
            COMMENT_LIKE_ADD, COMMENT_LIKE_DELETE, COMMENT_DISLIKE_ADD, COMMENT_DISLIKE_DELETE,
            FS_STORE_PHOTOUNLIM
    );

    public final EventCategory category;

    private final SetF<Param> params;

    EventType(EventCategory category, Param... params) {
        this.category = category;
        this.params = Cf.set(params);
    }

    public boolean isGroup() {
        return GROUP_TYPES.containsTs(this);
    }

    public boolean reject() {
        return params.containsTs(Param.REJECT);
    }

    @Override
    public String value() {
        return EnumUtils.toXmlName(this);
    }

    enum Param {
        REJECT
    }

    public static final StringEnumResolver<EventType> R = StringEnumResolver.r(EventType.class);
}
