package ru.yandex.chemodan.eventlog.events;

import java.util.regex.Pattern;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.regex.Matcher2;
import ru.yandex.misc.regex.Pattern2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public class MpfsPath extends DefaultObject {
    private static final Pattern2 PATTERN = Pattern2.compile("/([^/]+)(/.*)?", Pattern.DOTALL);

    private static final char PATH_DELIMITER_CHAR = '/';

    private static final String PATH_DELIMITER = Character.valueOf(PATH_DELIMITER_CHAR).toString();

    private static final String ROOT_RELATIVE_PATH = PATH_DELIMITER;

    @BenderIgnore
    public final String service;

    @BenderIgnore
    public final String relativePath;

    @BenderIgnore
    public final Option<ResourceType> resourceTypeO;

    @BenderPart(name="path", strictName = true)
    public final String value;

    MpfsPath(String service, String relativePath, ResourceType resourceType) {
        this(service, relativePath, Option.of(resourceType));
    }

    MpfsPath(String service, String relativePath, Option<ResourceType> resourceTypeO) {
        this.service = service;
        this.relativePath = relativePath;
        this.value = PATH_DELIMITER + service + relativePath;
        this.resourceTypeO = resourceTypeO;
    }

    private MpfsPath withRelativePath(String relativePath) {
        return new MpfsPath(service, relativePath, resourceTypeO);
    }

    public static MpfsPath parseFile(String value) {
        return parse(value, ResourceType.FILE);
    }

    public static MpfsPath parseDir(String value) {
        return parse(value, ResourceType.DIRECTORY);
    }

    public static MpfsPath parse(String value, ResourceType resourceType) {
        return parse(value, Option.of(resourceType));
    }

    public static MpfsPath parse(String value, Option<ResourceType> resourceTypeO) {
        Matcher2 matcher = PATTERN.matcher2(value);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("value = " + value + " doesn't match pattern = " + PATTERN);
        }

        String service = matcher.group(1).get();
        String relativePath = matcher.group(2).getOrElse("");
        return new MpfsPath(service, relativePath, resourceTypeO).sanitize();
    }

    private MpfsPath sanitize() {
        if (isRoot()) {
            return getRoot();
        } else if (resourceTypeO.isSome(ResourceType.DIRECTORY) && !hasSlashAtEnd()) {
            return withRelativePath(relativePath + PATH_DELIMITER);
        } else if (resourceTypeO.isSome(ResourceType.FILE) && hasSlashAtEnd()) {
            return withRelativePath(relativePath.substring(0, relativePath.length() - 1));
        } else {
            return this;
        }
    }

    private MpfsPath getRoot() {
        return getParent(ROOT_RELATIVE_PATH);
    }

    public String getName() {
        return relativePath.substring(nameOffset(), relativePath.length() - (hasSlashAtEnd() ? 1 : 0));
    }

    public MpfsPath getParent() {
        if (isRoot()) {
            return this;
        }

        return getParent(relativePath.substring(0, nameOffset()));
    }

    private MpfsPath getParent(String relativePath) {
        return new MpfsPath(service, relativePath, ResourceType.DIRECTORY);
    }

    public ListF<MpfsPath> getParents() {
        ListF<MpfsPath> result = Cf.arrayList();

        for (MpfsPath path = this; !path.isRoot(); ) {
            path = path.getParent();
            result.add(path);
        }
        return result;
    }

    public int getLevel() {
        return isRoot() ? 0 : StringUtils.countMatches(relativePath, PATH_DELIMITER) - 1;
    }

    public boolean isRoot() {
        return relativePath.isEmpty() || relativePath.equals(ROOT_RELATIVE_PATH);
    }

    public MpfsPath getDirectory() {
        return resourceTypeO.isSome(ResourceType.DIRECTORY) ? this : getParent();
    }

    public boolean startsWith(MpfsPath other) {
        return value.startsWith(other.value);
    }

    public String getValue() {
        return value;
    }

    @Override
    public String toString() {
        return value;
    }

    private int nameOffset() {
        return relativePath.lastIndexOf(PATH_DELIMITER_CHAR, relativePath.length() - (hasSlashAtEnd() ? 2 : 0)) + 1;
    }

    private boolean hasSlashAtEnd() {
        return !relativePath.isEmpty() && relativePath.charAt(relativePath.length() - 1) == PATH_DELIMITER_CHAR;
    }
}
