package ru.yandex.chemodan.eventlog.events.client;

import lombok.Data;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.parse.BenderParser;

@Data
public class MetricsEvent {

    public static final String EVENT_VALUE_FIELD = "EventValue";

    public static final String API_KEY_FIELD = "APIKey";

    public static final String ACCOUNT_ID_FIELD = "AccountID";

    private static final BenderParser<EventValue> EVENT_VALUE_PARSER = Bender.parser(EventValue.class);

    private final String apiKey;

    private final String appPlatform;

    private final Option<EventValue> eventValue;

    private final Instant timestamp;

    private final long accountId;

    private final String eventName;

    protected MetricsEvent(String apiKey, String appPlatform, Option<EventValue> eventValue,
            Instant timestamp, long accountId, String eventName)
    {
        this.apiKey = apiKey;
        this.appPlatform = appPlatform;
        this.eventValue = eventValue;
        this.timestamp = timestamp;
        this.accountId = accountId;
        this.eventName = eventName;
    }

    public static MetricsEvent parse(TskvEventLogLine logLine) {
        return new MetricsEvent(logLine.getString(API_KEY_FIELD), logLine.getString("AppPlatform"),
                logLine.getNonEmptyStringO(EVENT_VALUE_FIELD).map(EVENT_VALUE_PARSER::parseJson),
                new Instant(logLine.getBigInteger("EventTimestamp").longValue() * 1000L),
                Long.parseLong(logLine.getNonEmptyString(ACCOUNT_ID_FIELD)), logLine.getString("EventName"));
    }

    public MetricsEventType getType() {
        if (isOpenLentaTab()) {
            return MetricsEventType.OPEN_LENTA_TAB;
        }
        return MetricsEventType.UNKNOWN;
    }

    private boolean isOpenLentaTab() {
        if (!getEventValue().isPresent()) {
            return false;
        }
        if ("android".equals(getAppPlatform())) {
            return isOpenLentaTabInAndroid(getEventValue().get());
        }
        if ("iOS".equals(getAppPlatform())) {
            return isOpenLentaTabInIos(getEventValue().get());
        }
        return false;
    }

    private boolean isOpenLentaTabInAndroid(EventValue eventValue) {
        if (!"fragment_activity".equals(getEventName())) {
            return false;
        }
        return eventValue.getMainActivity().filter("FeedPartition"::equals).isPresent();
    }

    private boolean isOpenLentaTabInIos(EventValue eventValue) {
        if (!"feed".equals(getEventName())) {
            return false;
        }
        return eventValue.getMetrics().filter(metrics -> metrics.getTabOpen().isPresent()).isPresent() ||
                eventValue.getUi().filter("scroll"::equals).isPresent();
    }
}
