package ru.yandex.chemodan.eventlog.events.client;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.log.TskvClientMetricsEventLogLine;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

public class MetricsEventTest {

    private static final String IOS_OPEN_TAB_EVENT = "ADVID=\tAPIKey=18895\tAPIKey128=a7d6c160-30dd-47e0-8fe0-519653b50a08\t" +
            "AccountID=613694217\tAccountOptions_Keys=\tAccountOptions_Values=\tAccountType=login\tAge=25\tAndroidID=\t" +
            "AppBuildNumber=14062\tAppDebuggable=false\tAppFramework=NATIVE\tAppID=ru.yandex.disk\tAppPlatform=iOS\t" +
            "AppVersionName=256\tAttributionID=\tAttributionIDChanged=\tAttributionIDUUIDHash=7729261272165726177\t" +
            "CellID=\tCells_AreConnected=\tCells_CellsIDs=\tCells_CountriesCodes=\tCells_Lacs=\tCells_OperatorsIDs=\t" +
            "Cells_OperatorsNames=\tCells_PhysicalsCellsIDs=\tCells_SignalsStrengths=\tCells_Types=\tClickDate=1970-01-01\t" +
            "ClickDateTime=\tClickTimestamp=\tClids_Names=\tClids_Values=\tClientIP=::ffff:77.222.107.232\t" +
            "ClientIPHash=3352637382128983964\tClientKitVersion=298\tClientPort=50898\tConnectionType=CONN_WIFI\t" +
            "CountryCode=250\tCrashGroupID=\tCrashID=\tCustomAttribute_BoolValues=\tCustomAttribute_IdSetFlags=\t" +
            "CustomAttribute_Ids=\tCustomAttribute_NumberValues=\tCustomAttribute_ResetFlags=\tCustomAttribute_StringValues=\t" +
            "CustomAttribute_Types=\tCustomAttribute_Versions=\tDeduplicationEnabled=1\t" +
            "DeviceID=00000BDB-EBD5-45AD-820E-03481DAA3E4A\tDeviceIDHash=7746685741903524002\t" +
            "DeviceIDSessionIDHash=10744115796070672062\tDeviceType=PHONE\tErrorGroupID=\tErrorID=\tEventBytesTruncated=\t" +
            "EventDate=2019-09-15\tEventDateTime=2019-09-15 12:04:52\tEventEnvironment=\tEventEnvironmentParsedParams_Key1=\t" +
            "EventEnvironmentParsedParams_Key2=\tEventFirstOccurrence=undefined\tEventGlobalNumber=\tEventID=11003290636272127102\t" +
            "EventIsTruncated=\tEventName=feed\tEventNumber=8\tEventNumberOfType=\tEventSource=sdk\tEventTimeOffset=1\t" +
            "EventTimeZone=18000\tEventTimestamp=1568538292\tEventType=EVENT_CLIENT\t" +
            "EventValue={\\\"metrics\\\":{\\\"tab_open\\\":{\\\"timestamp\\\":\\\"2019-09-15T14:04:53.869+05:00\\\"}}}\t" +
            "EventValueCrash=\tEventValueError=\tEventValueJsonReference={0:{0:{0:\\\"0\\\"}}}\tEventValueReferrer=\t" +
            "IFV=00000BDB-EBD5-45AD-820E-03481DAA3E4A\tIFVHash=\tImei=\tIsProfileEventForReports=\tIsReengagement=\t" +
            "IsRevenueVerified=undefined\tIsRooted=\tKitBuildNumber=11459\tKitBuildType=102\tKitVersion=298\tLAC=\t" +
            "Latitude=55.176841735839844\tLimitAdTracking=false\tLocale=ru_RU\tLocationAltitude=\tLocationDirection=\t" +
            "LocationEnabled=true\tLocationPrecision=140\tLocationSource=LBS\tLocationSpeed=\tLocationTimestamp=\t" +
            "Longitude=61.522315979003906\tManufacturer=Apple\tModel=iPhone 5s\tNetworkType=HSDPA\tNetworksInterfaces_Macs=\t" +
            "NetworksInterfaces_Names=\tOSApiLevel=12\tOSVersion=12.4.1\tOpenType=\tOperatingSystem=ios\tOperatorID=20\t" +
            "OperatorName=Tele2\tOriginalDeviceID=7804B005-1ED7-43E4-BE31-37DEA840CDF1\tOriginalManufacturer=Apple\t" +
            "OriginalModel=iPhone6,2\tParsedParams_Key1=[\\\"metrics\\\"]\tParsedParams_Key10=[\\\"\\\"]\t" +
            "ParsedParams_Key2=[\\\"tab_open\\\"]\tParsedParams_Key3=[\\\"timestamp\\\"]\t" +
            "ParsedParams_Key4=[\\\"2019-09-15T14:04:53.869+05:00\\\"]\tParsedParams_Key5=[\\\"\\\"]\tParsedParams_Key6=[\\\"\\\"]\t" +
            "ParsedParams_Key7=[\\\"\\\"]\tParsedParams_Key8=[\\\"\\\"]\tParsedParams_Key9=[\\\"\\\"]\tParsedParams_Quantity=[1]\t" +
            "ParsedParams_ValueDouble=[0]\tProfileAttributeVersion=26315705712377864\tProfileID=\tProfileIDHash=\tPublisherID=\t" +
            "PushActionType=\tPushAndroidChannel=\tPushAndroidChannelEnabled=\tPushAndroidChannelGroup=\tPushAndroidGroup=\t" +
            "PushAndroidGroupEnabled=\tPushCampaignID=\tPushCorrelationID=\tPushEnabled=\tPushGroupID=\tPushHypothesisID=\t" +
            "PushIosAlertEnabled=\tPushIosBadgeEnabled=\tPushIosSoundEnabled=\tPushMessageID=\tPushTag=\tPushTransferID=\t" +
            "ReceiveDate=2019-09-15\tReceiveTimeZone=\tReceiveTimestamp=1568538294\tReferrerClickTimestamp=\t" +
            "ReferrerInstallationBeginTimestamp=\tRegionID=108161\tRegionTimeZone=18000\tReportEnvironment_Keys=[\\\"experiments\\\"]\t" +
            "ReportEnvironment_Values=[\\\"[89437,89426,100865,100342,94548,84242]\\\"]\tRequestID=\tRevenueCurrency=\t" +
            "RevenueOrderId=\tRevenueOrderIdHash=\tRevenueOrderIdSource=autogenerated\tRevenuePrice=\tRevenueProductId=\t" +
            "RevenueQuantity=\tScaleFactor=2\tScreenDPI=326\tScreenHeight=320\tScreenWidth=568\tSendTimeZone=18000\t" +
            "SendTimestamp=1568538294\tSessionID=1568538291\tSessionType=SESSION_BACKGROUND\tSex=2\tSignalStrength=\t" +
            "SimCards_AreRoaming=[0]\tSimCards_CountriesCodes=[250]\tSimCards_IccIDs=[\\\"\\\"]\tSimCards_OperatorsIDs=[20]\t" +
            "SimCards_OperatorsNames=[\\\"Tele2\\\"]\tStartDate=2019-09-15\tStartTime=2019-09-15 12:04:51\tStartTimeCorrected=\t" +
            "StartTimeObtainedBeforeSynchronization=\tStartTimeZone=18000\tStartTimestamp=1568538291\tTrackingID=\t" +
            "UUID=a387a9190f1dad3bd97f03a7879eb033\tUUIDHash=15873955353521992593\tUrlHost=\tUrlParameters_Keys=\t" +
            "UrlParameters_Values=\tUrlPath=\tUrlScheme=\tWifiAccessPointSsid=\tWifiAccessPointState=UNKNOWN\tWifi_AreConnected=[1]\t" +
            "Wifi_LastVisibleTimeOffset=[0]\tWifi_Macs=[\\\"E8DE27E54F82\\\"]\tWifi_SignalsStrengths=[0]\t" +
            "Wifi_Ssids=[\\\"TP-LINK_E54F82\\\"]\tWindowsAID=\tYMTrackingID=\t_logfeller_icookie=4231365450102547047\t" +
            "_logfeller_index_bucket=//home/logfeller/index/mobmetrika/metrika-mobile-log/900-1800/1568538600/1568538000\t" +
            "_logfeller_timestamp=1568538294\t_stbx=rt3.man--mobmetrika--metrika-mobile-log:69@@306260647@@GUQjXvvW4PZCmtbAkt5QRw@@1568538425079@@1568538461@@metrika-mobile-log@@642099880@@1568538448516\t" +
            "iso_eventtime=2019-09-15 12:04:54\tp2uid=\tp2uid_ses_type=\tpuid=\tpuid_ses_type=\t" +
            "source_uri=prt://mobmetrika@mtcalclog07k.yandex.ru/opt/statbox_export_mobile_events/move\tspuid=\tspuid_ses_type=\t" +
            "subkey=\ttimestamp=2019-09-15 09:04:54\ttimezone=+0000\ttskv_format=metrika-mobile-log";

    private static final String IOS_UI_SCROLL_EVENT = "ADVID=\tAPIKey=18895\tAPIKey128=a7d6c160-30dd-47e0-8fe0-519653b50a08\t" +
            "AccountID=551130820\tAccountOptions_Keys=\tAccountOptions_Values=\tAccountType=login\tAge=\tAndroidID=\t" +
            "AppBuildNumber=18177\tAppDebuggable=false\tAppFramework=NATIVE\tAppID=ru.yandex.disk\tAppPlatform=iOS\t" +
            "AppVersionName=282\tAttributionID=1\tAttributionIDChanged=\tAttributionIDUUIDHash=7782600639229135842\tCellID=\t" +
            "Cells_AreConnected=\tCells_CellsIDs=\tCells_CountriesCodes=\tCells_Lacs=\tCells_OperatorsIDs=\tCells_OperatorsNames=\t" +
            "Cells_PhysicalsCellsIDs=\tCells_SignalsStrengths=\tCells_Types=\tClickDate=1970-01-01\tClickDateTime=\tClickTimestamp=\t" +
            "Clids_Names=\tClids_Values=\tClientIP=::ffff:212.106.58.202\tClientIPHash=641252921069656583\tClientKitVersion=3007001\t" +
            "ClientPort=54407\tConnectionType=CONN_WIFI\tCountryCode=250\tCrashGroupID=\tCrashID=\tCustomAttribute_BoolValues=\t" +
            "CustomAttribute_IdSetFlags=\tCustomAttribute_Ids=\tCustomAttribute_NumberValues=\tCustomAttribute_ResetFlags=\t" +
            "CustomAttribute_StringValues=\tCustomAttribute_Types=\tCustomAttribute_Versions=\tDeduplicationEnabled=1\t" +
            "DeviceID=00007D2C-782B-4734-9072-37A7C7E4A40E\tDeviceIDHash=1743495795232237684\tDeviceIDSessionIDHash=14490732394459133548\t" +
            "DeviceType=PHONE\tErrorGroupID=\tErrorID=\tEventBytesTruncated=\tEventDate=2019-09-14\tEventDateTime=2019-09-14 04:53:58\t" +
            "EventEnvironment=\tEventEnvironmentParsedParams_Key1=\tEventEnvironmentParsedParams_Key2=\tEventFirstOccurrence=false\t" +
            "EventGlobalNumber=52096\tEventID=8624232323659932559\tEventIsTruncated=\tEventName=feed\tEventNumber=925\t" +
            "EventNumberOfType=50444\tEventSource=sdk\tEventTimeOffset=56\tEventTimeZone=10800\tEventTimestamp=1568426038\t" +
            "EventType=EVENT_CLIENT\tEventValue={\\\"ui\\\":\\\"scroll\\\"}\tEventValueCrash=\tEventValueError=\tEventValueJsonReference={0:\\\"0\\\"}\t" +
            "EventValueReferrer=\tIFV=00007D2C-782B-4734-9072-37A7C7E4A40E\tIFVHash=\tImei=\tIsProfileEventForReports=\t" +
            "IsReengagement=\tIsRevenueVerified=undefined\tIsRooted=\tKitBuildNumber=15098\tKitBuildType=101\tKitVersion=3007001\t" +
            "LAC=\tLatitude=\tLimitAdTracking=false\tLocale=ru_RU\tLocationAltitude=\tLocationDirection=\tLocationEnabled=true\t" +
            "LocationPrecision=\tLocationSource=MISSING\tLocationSpeed=\tLocationTimestamp=\tLongitude=\tManufacturer=Apple\t" +
            "Model=iPhone 5s\tNetworkType=LTE\tNetworksInterfaces_Macs=\tNetworksInterfaces_Names=\tOSApiLevel=12\tOSVersion=12.4\t" +
            "OpenType=\tOperatingSystem=ios\tOperatorID=1\tOperatorName=Mobile TeleSystems\tOriginalDeviceID=078F9D10-EAE7-4EB6-891F-153A42A16C06\t" +
            "OriginalManufacturer=Apple\tOriginalModel=iPhone6,2\tParsedParams_Key1=[\\\"ui\\\"]\tParsedParams_Key10=[\\\"\\\"]\t" +
            "ParsedParams_Key2=[\\\"scroll\\\"]\tParsedParams_Key3=[\\\"\\\"]\tParsedParams_Key4=[\\\"\\\"]\tParsedParams_Key5=[\\\"\\\"]\t" +
            "ParsedParams_Key6=[\\\"\\\"]\tParsedParams_Key7=[\\\"\\\"]\tParsedParams_Key8=[\\\"\\\"]\tParsedParams_Key9=[\\\"\\\"]\t" +
            "ParsedParams_Quantity=[1]\tParsedParams_ValueDouble=[0]\tProfileAttributeVersion=26313821480027037\tProfileID=\t" +
            "ProfileIDHash=\tPublisherID=\tPushActionType=\tPushAndroidChannel=\tPushAndroidChannelEnabled=\tPushAndroidChannelGroup=\t" +
            "PushAndroidGroup=\tPushAndroidGroupEnabled=\tPushCampaignID=\tPushCorrelationID=\tPushEnabled=\tPushGroupID=\t" +
            "PushHypothesisID=\tPushIosAlertEnabled=\tPushIosBadgeEnabled=\tPushIosSoundEnabled=\tPushMessageID=\tPushTag=\t" +
            "PushTransferID=\tReceiveDate=2019-09-15\tReceiveTimeZone=\tReceiveTimestamp=1568550543\tReferrerClickTimestamp=\t" +
            "ReferrerInstallationBeginTimestamp=\tRegionID=967\tRegionTimeZone=10800\tReportEnvironment_Keys=[\\\"uid\\\",\\\"experiments\\\"]\t" +
            "ReportEnvironment_Values=[\\\"551130820\\\",\\\"[169564,151723,156100,166693,169238,144621,161179]\\\"]\t" +
            "RequestID=5860\tRevenueCurrency=\tRevenueOrderId=\tRevenueOrderIdHash=\tRevenueOrderIdSource=autogenerated\t" +
            "RevenuePrice=\tRevenueProductId=\tRevenueQuantity=\tScaleFactor=2\tScreenDPI=326\tScreenHeight=320\tScreenWidth=568\t" +
            "SendTimeZone=\tSendTimestamp=1568550542\tSessionID=10000002625\tSessionType=SESSION_FOREGROUND\tSex=\tSignalStrength=\t" +
            "SimCards_AreRoaming=[0]\tSimCards_CountriesCodes=[250]\tSimCards_IccIDs=[\\\"\\\"]\tSimCards_OperatorsIDs=[1]\t" +
            "SimCards_OperatorsNames=[\\\"Mobile TeleSystems\\\"]\tStartDate=2019-09-14\tStartTime=2019-09-14 04:53:02\t" +
            "StartTimeCorrected=\tStartTimeObtainedBeforeSynchronization=\tStartTimeZone=10800\tStartTimestamp=1568425982\t" +
            "TrackingID=\tUUID=2deea70cbb71b0116d18b20b1a569684\tUUIDHash=11635288358661503094\tUrlHost=\tUrlParameters_Keys=\t" +
            "UrlParameters_Values=\tUrlPath=\tUrlScheme=\tWifiAccessPointSsid=\tWifiAccessPointState=UNKNOWN\tWifi_AreConnected=\t" +
            "Wifi_LastVisibleTimeOffset=\tWifi_Macs=\tWifi_SignalsStrengths=\tWifi_Ssids=\tWindowsAID=\tYMTrackingID=\t" +
            "_logfeller_icookie=9859384401027704042\t_logfeller_index_bucket=//home/logfeller/index/mobmetrika/metrika-mobile-log/900-1800/1568550300/1568548800\t" +
            "_logfeller_timestamp=1568550543\t_stbx=rt3.sas--mobmetrika--metrika-mobile-log:40@@258943274@@jWYvQPLCVdkA1_6liCCwUg@@1568550692581@@1568550717@@metrika-mobile-log@@531907449@@1568550716962\tiso_eventtime=2019-09-15 15:29:03\t" +
            "p2uid=\tp2uid_ses_type=\tpuid=\tpuid_ses_type=\tsource_uri=prt://mobmetrika@mtcalclog34i.metrika.yandex.net/opt/statbox_export_mobile_events/move\t" +
            "spuid=\tspuid_ses_type=\tsubkey=\ttimestamp=2019-09-15 12:29:03\ttimezone=+0000\ttskv_format=metrika-mobile-log";

    private static final String ANDROID_OPEN_TAB_EVENT = "ADVID=36627f77-2a08-4d79-8cac-f976ae529d89\tAPIKey=18895\t" +
            "APIKey128=a7d6c160-30dd-47e0-8fe0-519653b50a08\tAccountID=803496840\tAccountOptions_Keys=\tAccountOptions_Values=\t" +
            "AccountType=\tAge=35\tAndroidID=4506192393bc425b\tAppBuildNumber=418020\tAppDebuggable=false\tAppFramework=NATIVE\t" +
            "AppID=ru.yandex.disk\tAppPlatform=android\tAppVersionName=4.18.2\tAttributionID=1\tAttributionIDChanged=\t" +
            "AttributionIDUUIDHash=9368991777645335950\tCellID=\tCells_AreConnected=[1]\tCells_CellsIDs=[141674804]\t" +
            "Cells_CountriesCodes=[250]\tCells_Lacs=[12606]\tCells_OperatorsIDs=[1]\tCells_OperatorsNames=[\\\"MTS RUS\\\"]\t" +
            "Cells_PhysicalsCellsIDs=[0]\tCells_SignalsStrengths=[-111]\tCells_Types=[DEFAULT]\tClickDate=1970-01-01\t" +
            "ClickDateTime=\tClickTimestamp=\tClids_Names=\tClids_Values=\tClientIP=::ffff:94.233.244.211\t" +
            "ClientIPHash=9596787964642239618\tClientKitVersion=3006004\tClientPort=1428\tConnectionType=CONN_WIFI\t" +
            "CountryCode=250\tCrashGroupID=\tCrashID=\tCustomAttribute_BoolValues=\tCustomAttribute_IdSetFlags=\t" +
            "CustomAttribute_Ids=\tCustomAttribute_NumberValues=\tCustomAttribute_ResetFlags=\tCustomAttribute_StringValues=\t" +
            "CustomAttribute_Types=\tCustomAttribute_Versions=\tDeduplicationEnabled=1\tDeviceID=0001a19f3621553a3ca6259037ece311\t" +
            "DeviceIDHash=903248915045928333\tDeviceIDSessionIDHash=13176986596242872484\tDeviceType=PHONE\tErrorGroupID=\t" +
            "ErrorID=\tEventBytesTruncated=\tEventDate=2019-09-15\tEventDateTime=2019-09-15 15:20:22\tEventEnvironment=\t" +
            "EventEnvironmentParsedParams_Key1=\tEventEnvironmentParsedParams_Key2=\tEventFirstOccurrence=false\tEventGlobalNumber=71090\t" +
            "EventID=7418878478027717130\tEventIsTruncated=\tEventName=fragment_activity\tEventNumber=8\tEventNumberOfType=9440\t" +
            "EventSource=sdk\tEventTimeOffset=\tEventTimeZone=10800\tEventTimestamp=1568550022\tEventType=EVENT_CLIENT\t" +
            "EventValue={\\\"MainActivity\\\":\\\"FeedPartition\\\"}\tEventValueCrash=\tEventValueError=\t" +
            "EventValueJsonReference={0:\\\"0\\\"}\tEventValueReferrer=\tIFV=\tIFVHash=\tImei=\tIsProfileEventForReports=\t" +
            "IsReengagement=\tIsRevenueVerified=undefined\tIsRooted=\tKitBuildNumber=45178\tKitBuildType=202\tKitVersion=3006004\t" +
            "LAC=\tLatitude=45.44144144144144\tLimitAdTracking=false\tLocale=ru_RU\tLocationAltitude=\tLocationDirection=\t" +
            "LocationEnabled=true\tLocationPrecision=2000\tLocationSource=PLATFORM_LBS\tLocationSpeed=\tLocationTimestamp=1568549814\t" +
            "Longitude=41.036543339463755\tManufacturer=Samsung\tModel=Galaxy J3(2016)\tNetworkType=LTE\t" +
            "NetworksInterfaces_Macs=[\\\"0E1D422B93FC\\\",\\\"0E753D237E98\\\",\\\"CE3B3D6FCFE2\\\",\\\"D679B9D60886\\\"," +
            "\\\"2A87AFE0A7A9\\\",\\\"C69DF6460D47\\\",\\\"DE729CC00EDE\\\",\\\"2EA6370D46DA\\\",\\\"7C2EDDB2EE4F\\\"," +
            "\\\"7E2EDDB2EE4F\\\"]\tNetworksInterfaces_Names=[\\\"rmnet0\\\",\\\"rmnet1\\\",\\\"rmnet5\\\",\\\"rmnet6\\\"," +
            "\\\"rmnet7\\\",\\\"rmnet2\\\",\\\"rmnet3\\\",\\\"rmnet4\\\",\\\"wlan0\\\",\\\"p2p0\\\"]\tOSApiLevel=22\tOSVersion=5.1.1\t" +
            "OpenType=\tOperatingSystem=android\tOperatorID=1\tOperatorName=MTS RUS\tOriginalDeviceID=\tOriginalManufacturer=samsung\t" +
            "OriginalModel=SM-J320F\tParsedParams_Key1=[\\\"MainActivity\\\"]\tParsedParams_Key10=[\\\"\\\"]\t" +
            "ParsedParams_Key2=[\\\"FeedPartition\\\"]\tParsedParams_Key3=[\\\"\\\"]\tParsedParams_Key4=[\\\"\\\"]\t" +
            "ParsedParams_Key5=[\\\"\\\"]\tParsedParams_Key6=[\\\"\\\"]\tParsedParams_Key7=[\\\"\\\"]\tParsedParams_Key8=[\\\"\\\"]\t" +
            "ParsedParams_Key9=[\\\"\\\"]\tParsedParams_Quantity=[1]\tParsedParams_ValueDouble=[0]\t" +
            "ProfileAttributeVersion=26315902525898760\tProfileID=\tProfileIDHash=\tPublisherID=\tPushActionType=\tPushAndroidChannel=\t" +
            "PushAndroidChannelEnabled=\tPushAndroidChannelGroup=\tPushAndroidGroup=\tPushAndroidGroupEnabled=\tPushCampaignID=\t" +
            "PushCorrelationID=\tPushEnabled=\tPushGroupID=\tPushHypothesisID=\tPushIosAlertEnabled=\tPushIosBadgeEnabled=\t" +
            "PushIosSoundEnabled=\tPushMessageID=\tPushTag=\tPushTransferID=\tReceiveDate=2019-09-15\tReceiveTimeZone=\t" +
            "ReceiveTimestamp=1568550024\tReferrerClickTimestamp=\tReferrerInstallationBeginTimestamp=\tRegionID=139944\t" +
            "RegionTimeZone=10800\tReportEnvironment_Keys=[\\\"experiments\\\"]\tReportEnvironment_Values=[\\\"[\\\\\\\"136290\\\\\\\"," +
            "\\\\\\\"142593\\\\\\\",\\\\\\\"144621\\\\\\\",\\\\\\\"151723\\\\\\\",\\\\\\\"156100\\\\\\\",\\\\\\\"166406\\\\\\\"," +
            "\\\\\\\"166693\\\\\\\",\\\\\\\"169564\\\\\\\"]\\\"]\tRequestID=3799\tRevenueCurrency=\tRevenueOrderId=\t" +
            "RevenueOrderIdHash=\tRevenueOrderIdSource=autogenerated\tRevenuePrice=\tRevenueProductId=\tRevenueQuantity=\t" +
            "ScaleFactor=2\tScreenDPI=320\tScreenHeight=720\tScreenWidth=1280\tSendTimeZone=\tSendTimestamp=1568550023\t" +
            "SessionID=10000000323\tSessionType=SESSION_FOREGROUND\tSex=1\tSignalStrength=\tSimCards_AreRoaming=[0]\t" +
            "SimCards_CountriesCodes=[250]\tSimCards_IccIDs=[\\\"\\\"]\tSimCards_OperatorsIDs=[1]\tSimCards_OperatorsNames=[\\\"MTS RUS\\\"]\t" +
            "StartDate=2019-09-15\tStartTime=2019-09-15 15:20:22\tStartTimeCorrected=\tStartTimeObtainedBeforeSynchronization=\t" +
            "StartTimeZone=10800\tStartTimestamp=1568550022\tTrackingID=\tUUID=d2579f61b3fc433f869a10f90a2d1119\t" +
            "UUIDHash=17105571780610037671\tUrlHost=\tUrlParameters_Keys=\tUrlParameters_Values=\tUrlPath=\tUrlScheme=\t" +
            "WifiAccessPointSsid=AndroidAP\tWifiAccessPointState=DISABLED\tWifi_AreConnected=[1]\tWifi_LastVisibleTimeOffset=[3]\t" +
            "Wifi_Macs=[\\\"10FEED7850DE\\\"]\tWifi_SignalsStrengths=[-70]\tWifi_Ssids=[\\\"TP-LINK_7850DE\\\"]\tWindowsAID=\t" +
            "YMTrackingID=\t_logfeller_icookie=2951526350465179875\t" +
            "_logfeller_index_bucket=//home/logfeller/index/mobmetrika/metrika-mobile-log/900-1800/1568550300/1568548800\t" +
            "_logfeller_timestamp=1568550024\t_stbx=rt3.man--mobmetrika--metrika-mobile-log:47@@308772183@@G8fPDLA1QuALCdlpow-r-g@@1568550141300@@1568550172@@metrika-mobile-log@@852090675@@1568550171024\t" +
            "iso_eventtime=2019-09-15 15:20:24\tp2uid=\tp2uid_ses_type=\tpuid=\tpuid_ses_type=\t" +
            "source_uri=prt://mobmetrika@mtcalclog54k.metrika.yandex.net/opt/statbox_export_mobile_events/move\tspuid=\t" +
            "spuid_ses_type=\tsubkey=\ttimestamp=2019-09-15 12:20:24\ttimezone=+0000\ttskv_format=metrika-mobile-log";

    private static final String EVENT_CLIENT_NON_LENTA_TAB_OPEN = "ADVID=d1884a2a-26ad-4cec-a884-66932d33e327\tAPIKey=1\t" +
            "APIKey128=a411ca3d-6d98-4c96-9ab2-25ee2fcdffdf\tAccountID=803496840\tAccountOptions_Keys=\tAccountOptions_Values=\t" +
            "AccountType=\tAge=55\tAndroidID=538fde57b346640e\tAppBuildNumber=18951\tAppDebuggable=undefined\t" +
            "AppFramework=NATIVE\tAppID=com.yandex.store\tAppPlatform=android\tAppVersionName=2.44\tAttributionID=\t" +
            "AttributionIDChanged=\tAttributionIDUUIDHash=13971580252449404899\tCellID=\tCells_AreConnected=[1]\t" +
            "Cells_CellsIDs=[102410000]\tCells_CountriesCodes=[250]\tCells_Lacs=[9203]\tCells_OperatorsIDs=[20]\t" +
            "Cells_OperatorsNames=[\\\"Tele2\\\"]\tCells_PhysicalsCellsIDs=[0]\tCells_SignalsStrengths=[-93]\t" +
            "Cells_Types=[DEFAULT]\tClickDate=1970-01-01\tClickDateTime=\tClickTimestamp=\tClids_Names=\tClids_Values=\t" +
            "ClientIP=::ffff:176.59.33.36\tClientIPHash=4959356722682527340\tClientKitVersion=261\tClientPort=50169\t" +
            "ConnectionType=CONN_CELL\tCountryCode=250\tCrashGroupID=\tCrashID=\tCustomAttribute_BoolValues=\t" +
            "CustomAttribute_IdSetFlags=\tCustomAttribute_Ids=\tCustomAttribute_NumberValues=\tCustomAttribute_ResetFlags=\t" +
            "CustomAttribute_StringValues=\tCustomAttribute_Types=\tCustomAttribute_Versions=\tDeduplicationEnabled=1\t" +
            "DeviceID=00042848a34576671647fcf85cf5eb62\tDeviceIDHash=9445970006957997630\tDeviceIDSessionIDHash=16687105292028422547\t" +
            "DeviceType=TABLET\tErrorGroupID=\tErrorID=\tEventBytesTruncated=\tEventDate=2019-09-15\tEventDateTime=2019-09-15 18:41:26\t" +
            "EventEnvironment=\tEventEnvironmentParsedParams_Key1=\tEventEnvironmentParsedParams_Key2=\tEventFirstOccurrence=undefined\t" +
            "EventGlobalNumber=\tEventID=8989925018287997247\tEventIsTruncated=\tEventName=app_removals\tEventNumber=4\t" +
            "EventNumberOfType=\tEventSource=sdk\tEventTimeOffset=\tEventTimeZone=10800\tEventTimestamp=1568562086\t" +
            "EventType=EVENT_CLIENT\tEventValue={\\\"billing_clid\\\":\\\"0\\\",\\\"system_store\\\":\\\"false\\\"," +
            "\\\"startup_clid\\\":\\\"0\\\"}\tEventValueCrash=\tEventValueError=\tEventValueJsonReference={0:\\\"0\\\"," +
            "1:\\\"1\\\",2:\\\"0\\\"}\tEventValueReferrer=\tIFV=\tIFVHash=\tImei=[\\\"866150021431164\\\"]\t" +
            "IsProfileEventForReports=\tIsReengagement=\tIsRevenueVerified=undefined\tIsRooted=\tKitBuildNumber=6314\t" +
            "KitBuildType=202\tKitVersion=261\tLAC=\tLatitude=55.1138394\tLimitAdTracking=false\tLocale=ru_RU\tLocationAltitude=\t" +
            "LocationDirection=\tLocationEnabled=undefined\tLocationPrecision=1899\tLocationSource=PLATFORM_LBS\tLocationSpeed=\t" +
            "LocationTimestamp=1568561979\tLongitude=36.5889665\tManufacturer=CUBE\tModel=T8PLUSML\tNetworkType=LTE\t" +
            "NetworksInterfaces_Macs=[\\\"460E7BF8C7C4\\\",\\\"4E3A8995385F\\\",\\\"7619D5BB03B4\\\",\\\"9E98806B8DF0\\\"," +
            "\\\"16B9BDDFD3AE\\\"]\tNetworksInterfaces_Names=[\\\"ccmni2\\\",\\\"ccmni0\\\",\\\"ccmni1\\\",\\\"ifb0\\\"," +
            "\\\"ifb1\\\"]\tOSApiLevel=22\tOSVersion=5.1\tOpenType=\tOperatingSystem=android\tOperatorID=20\tOperatorName=Tele2\t" +
            "OriginalDeviceID=\tOriginalManufacturer=CUBE\tOriginalModel=T8PLUSML\tParsedParams_Key1=[\\\"billing_clid\\\"," +
            "\\\"system_store\\\",\\\"startup_clid\\\"]\tParsedParams_Key10=[\\\"\\\",\\\"\\\",\\\"\\\"]\t" +
            "ParsedParams_Key2=[\\\"0\\\",\\\"false\\\",\\\"0\\\"]\tParsedParams_Key3=[\\\"\\\",\\\"\\\",\\\"\\\"]\t" +
            "ParsedParams_Key4=[\\\"\\\",\\\"\\\",\\\"\\\"]\tParsedParams_Key5=[\\\"\\\",\\\"\\\",\\\"\\\"]\t" +
            "ParsedParams_Key6=[\\\"\\\",\\\"\\\",\\\"\\\"]\tParsedParams_Key7=[\\\"\\\",\\\"\\\",\\\"\\\"]\t" +
            "ParsedParams_Key8=[\\\"\\\",\\\"\\\",\\\"\\\"]\tParsedParams_Key9=[\\\"\\\",\\\"\\\",\\\"\\\"]\t" +
            "ParsedParams_Quantity=[1,1,1]\tParsedParams_ValueDouble=[0,0,0]\tProfileAttributeVersion=26316104926232580\t" +
            "ProfileID=\tProfileIDHash=\tPublisherID=\tPushActionType=\tPushAndroidChannel=\tPushAndroidChannelEnabled=\t" +
            "PushAndroidChannelGroup=\tPushAndroidGroup=\tPushAndroidGroupEnabled=\tPushCampaignID=\tPushCorrelationID=\t" +
            "PushEnabled=\tPushGroupID=\tPushHypothesisID=\tPushIosAlertEnabled=\tPushIosBadgeEnabled=\tPushIosSoundEnabled=\t" +
            "PushMessageID=\tPushTag=\tPushTransferID=\tReceiveDate=2019-09-15\tReceiveTimeZone=\tReceiveTimestamp=1568562087\t" +
            "ReferrerClickTimestamp=\tReferrerInstallationBeginTimestamp=\tRegionID=217930\tRegionTimeZone=10800\t" +
            "ReportEnvironment_Keys=\tReportEnvironment_Values=\tRequestID=\tRevenueCurrency=\tRevenueOrderId=\t" +
            "RevenueOrderIdHash=\tRevenueOrderIdSource=autogenerated\tRevenuePrice=\tRevenueProductId=\tRevenueQuantity=\t" +
            "ScaleFactor=1.75\tScreenDPI=280\tScreenHeight=1200\tScreenWidth=1920\tSendTimeZone=10800\tSendTimestamp=1568562087\t" +
            "SessionID=1568562086\tSessionType=SESSION_BACKGROUND\tSex=1\tSignalStrength=\tSimCards_AreRoaming=[0]\t" +
            "SimCards_CountriesCodes=[250]\tSimCards_IccIDs=[\\\"\\\"]\tSimCards_OperatorsIDs=[20]\tSimCards_OperatorsNames=[\\\"Tele2\\\"]\t" +
            "StartDate=2019-09-15\tStartTime=2019-09-15 18:41:26\tStartTimeCorrected=\tStartTimeObtainedBeforeSynchronization=\t" +
            "StartTimeZone=10800\tStartTimestamp=1568562086\tTrackingID=\tUUID=935be5227266b62ec2385d438986c31d\t" +
            "UUIDHash=12617607447758256468\tUrlHost=\tUrlParameters_Keys=\tUrlParameters_Values=\tUrlPath=\tUrlScheme=\t" +
            "WifiAccessPointSsid=T8PLUSML\tWifiAccessPointState=DISABLED\tWifi_AreConnected=\tWifi_LastVisibleTimeOffset=\t" +
            "Wifi_Macs=\tWifi_SignalsStrengths=\tWifi_Ssids=\tWindowsAID=\tYMTrackingID=\t_logfeller_icookie=8908291151024067360\t" +
            "_logfeller_index_bucket=//home/logfeller/index/mobmetrika/metrika-mobile-log/900-1800/1568562900/1568561400\t" +
            "_logfeller_timestamp=1568562087\t_stbx=rt3.vla--mobmetrika--metrika-mobile-log:9@@330455584@@WLZVwKRq8Yq73dDrlTcmMg@@1568562544493@@1568562561@@metrika-mobile-log@@65167300@@1568562551037\tiso_eventtime=2019-09-15 18:41:27\t" +
            "p2uid=\tp2uid_ses_type=\tpuid=\tpuid_ses_type=\t" +
            "source_uri=prt://mobmetrika@mtcalclog68m.metrika.yandex.net/opt/statbox_export_mobile_events/move\t" +
            "spuid=\tspuid_ses_type=\tsubkey=\ttimestamp=2019-09-15 15:41:27\ttimezone=+0000\ttskv_format=metrika-mobile-log";

    private static final String NON_CLIENT_EVENT = "ADVID=d1884a2a-26ad-4cec-a884-66932d33e327\tAPIKey=1\t" +
            "APIKey128=a411ca3d-6d98-4c96-9ab2-25ee2fcdffdf\tAccountID=803496840\tAccountOptions_Keys=\tAccountOptions_Values=\t" +
            "AccountType=\tAge=55\tAndroidID=538fde57b346640e\tAppBuildNumber=18951\tAppDebuggable=undefined\tAppFramework=NATIVE\t" +
            "AppID=com.yandex.store\tAppPlatform=android\tAppVersionName=2.44\tAttributionID=\tAttributionIDChanged=\t" +
            "AttributionIDUUIDHash=13971580252449404899\tCellID=\tCells_AreConnected=[1]\tCells_CellsIDs=[102413837]\t" +
            "Cells_CountriesCodes=[250]\tCells_Lacs=[9208]\tCells_OperatorsIDs=[20]\tCells_OperatorsNames=[\\\"Tele2\\\"]\t" +
            "Cells_PhysicalsCellsIDs=[0]\tCells_SignalsStrengths=[0]\tCells_Types=[DEFAULT]\tClickDate=1970-01-01\t" +
            "ClickDateTime=\tClickTimestamp=\tClids_Names=\tClids_Values=\tClientIP=::ffff:176.59.33.36\tClientIPHash=4959356722682527340\t" +
            "ClientKitVersion=261\tClientPort=50169\tConnectionType=CONN_WIFI\tCountryCode=250\tCrashGroupID=\tCrashID=\t" +
            "CustomAttribute_BoolValues=\tCustomAttribute_IdSetFlags=\tCustomAttribute_Ids=\tCustomAttribute_NumberValues=\t" +
            "CustomAttribute_ResetFlags=\tCustomAttribute_StringValues=\tCustomAttribute_Types=\tCustomAttribute_Versions=\t" +
            "DeduplicationEnabled=1\tDeviceID=00042848a34576671647fcf85cf5eb62\tDeviceIDHash=9445970006957997630\t" +
            "DeviceIDSessionIDHash=4072671757373854085\tDeviceType=TABLET\tErrorGroupID=\tErrorID=\tEventBytesTruncated=\t" +
            "EventDate=2019-09-13\tEventDateTime=2019-09-13 11:04:39\tEventEnvironment={\\\"appmetrica_push_version\\\":\\\"4\\\"}\t" +
            "EventEnvironmentParsedParams_Key1=[\\\"appmetrica_push_version\\\"]\tEventEnvironmentParsedParams_Key2=[\\\"4\\\"]\t" +
            "EventFirstOccurrence=undefined\tEventGlobalNumber=\tEventID=7831071112442012632\tEventIsTruncated=\tEventName=\t" +
            "EventNumber=2\tEventNumberOfType=\tEventSource=sdk\tEventTimeOffset=13\tEventTimeZone=10800\tEventTimestamp=1568361879\t" +
            "EventType=EVENT_ALIVE\tEventValue=\tEventValueCrash=\tEventValueError=\tEventValueJsonReference=\tEventValueReferrer=\t" +
            "IFV=\tIFVHash=\tImei=[\\\"866150021431164\\\"]\tIsProfileEventForReports=\tIsReengagement=\tIsRevenueVerified=undefined\t" +
            "IsRooted=\tKitBuildNumber=6314\tKitBuildType=202\tKitVersion=261\tLAC=\tLatitude=55.116939544677734\tLimitAdTracking=false\t" +
            "Locale=ru_RU\tLocationAltitude=\tLocationDirection=\tLocationEnabled=undefined\tLocationPrecision=140\tLocationSource=LBS\t" +
            "LocationSpeed=\tLocationTimestamp=\tLongitude=36.5820426940918\tManufacturer=CUBE\tModel=T8PLUSML\tNetworkType=LTE\t" +
            "NetworksInterfaces_Macs=[\\\"460E7BF8C7C4\\\",\\\"4E3A8995385F\\\",\\\"7619D5BB03B4\\\",\\\"9E98806B8DF0\\\"," +
            "\\\"16B9BDDFD3AE\\\"]\tNetworksInterfaces_Names=[\\\"ccmni2\\\",\\\"ccmni0\\\",\\\"ccmni1\\\",\\\"ifb0\\\",\\\"ifb1\\\"]\t" +
            "OSApiLevel=22\tOSVersion=5.1\tOpenType=\tOperatingSystem=android\tOperatorID=20\tOperatorName=Tele2\tOriginalDeviceID=\t" +
            "OriginalManufacturer=CUBE\tOriginalModel=T8PLUSML\tParsedParams_Key1=\tParsedParams_Key10=\tParsedParams_Key2=\t" +
            "ParsedParams_Key3=\tParsedParams_Key4=\tParsedParams_Key5=\tParsedParams_Key6=\tParsedParams_Key7=\tParsedParams_Key8=\t" +
            "ParsedParams_Key9=\tParsedParams_Quantity=\tParsedParams_ValueDouble=\tProfileAttributeVersion=26312745792045058\t" +
            "ProfileID=\tProfileIDHash=\tPublisherID=\tPushActionType=\tPushAndroidChannel=\tPushAndroidChannelEnabled=\t" +
            "PushAndroidChannelGroup=\tPushAndroidGroup=\tPushAndroidGroupEnabled=\tPushCampaignID=\tPushCorrelationID=\t" +
            "PushEnabled=\tPushGroupID=\tPushHypothesisID=\tPushIosAlertEnabled=\tPushIosBadgeEnabled=\tPushIosSoundEnabled=\t" +
            "PushMessageID=\tPushTag=\tPushTransferID=\tReceiveDate=2019-09-15\tReceiveTimeZone=\tReceiveTimestamp=1568562087\t" +
            "ReferrerClickTimestamp=\tReferrerInstallationBeginTimestamp=\tRegionID=217928\tRegionTimeZone=10800\tReportEnvironment_Keys=\t" +
            "ReportEnvironment_Values=\tRequestID=\tRevenueCurrency=\tRevenueOrderId=\tRevenueOrderIdHash=\tRevenueOrderIdSource=autogenerated\t" +
            "RevenuePrice=\tRevenueProductId=\tRevenueQuantity=\tScaleFactor=1.75\tScreenDPI=280\tScreenHeight=1200\tScreenWidth=1920\t" +
            "SendTimeZone=10800\tSendTimestamp=1568562087\tSessionID=1568361866\tSessionType=SESSION_FOREGROUND\tSex=1\tSignalStrength=\t" +
            "SimCards_AreRoaming=[0]\tSimCards_CountriesCodes=[250]\tSimCards_IccIDs=[\\\"\\\"]\tSimCards_OperatorsIDs=[20]\t" +
            "SimCards_OperatorsNames=[\\\"Tele2\\\"]\tStartDate=2019-09-13\tStartTime=2019-09-13 11:04:26\tStartTimeCorrected=\t" +
            "StartTimeObtainedBeforeSynchronization=\tStartTimeZone=10800\tStartTimestamp=1568361866\tTrackingID=\t" +
            "UUID=935be5227266b62ec2385d438986c31d\tUUIDHash=12617607447758256468\tUrlHost=\tUrlParameters_Keys=\tUrlParameters_Values=\t" +
            "UrlPath=\tUrlScheme=\tWifiAccessPointSsid=T8PLUSML\tWifiAccessPointState=DISABLED\tWifi_AreConnected=[1,0,0,0,0,0,0,0,0,0,0,0]\t" +
            "Wifi_LastVisibleTimeOffset=[0,0,0,0,0,0,0,0,0,0,0,0]\tWifi_Macs=[\\\"4C60DED0A0D6\\\",\\\"34080436019B\\\"," +
            "\\\"E4186B03F6CC\\\",\\\"1C740D8DF388\\\",\\\"28285D636A16\\\",\\\"C025E90105BC\\\",\\\"28285D9F9C68\\\"," +
            "\\\"28285D6A169A\\\",\\\"0018E7FD69FC\\\",\\\"0E2C547DDD21\\\",\\\"48EE0CC73426\\\",\\\"04BF6D0D1260\\\"]\t" +
            "Wifi_SignalsStrengths=[-29,-57,-58,-78,-51,-69,-67,-85,-77,-83,-82,-86]\tWifi_Ssids=[\\\"admin\\\",\\\"Masha\\\"," +
            "\\\"Keenetic-0685\\\",\\\"buker63\\\",\\\"51kvartal\\\",\\\"RT_101\\\",\\\"Svolochi94\\\",\\\"Alexandr\\\"," +
            "\\\"ramZES\\\",\\\"HUAWEI Y5 Prime 2018\\\",\\\"TTK-HOME\\\",\\\"Keenetic-4215\\\"]\tWindowsAID=\tYMTrackingID=\t" +
            "_logfeller_icookie=8908291151024067360\t" +
            "_logfeller_index_bucket=//home/logfeller/index/mobmetrika/metrika-mobile-log/900-1800/1568562900/1568561400\t" +
            "_logfeller_timestamp=1568562087\t" +
            "_stbx=rt3.vla--mobmetrika--metrika-mobile-log:9@@330455584@@WLZVwKRq8Yq73dDrlTcmMg@@1568562544493@@1568562561@@metrika-mobile-log@@65167300@@1568562551037\t" +
            "iso_eventtime=2019-09-15 18:41:27\tp2uid=\tp2uid_ses_type=\tpuid=\tpuid_ses_type=\t" +
            "source_uri=prt://mobmetrika@mtcalclog68m.metrika.yandex.net/opt/statbox_export_mobile_events/move\t" +
            "spuid=\tspuid_ses_type=\tsubkey=\ttimestamp=2019-09-15 15:41:27\ttimezone=+0000\ttskv_format=metrika-mobile-log";

    @Test
    public void testIosOpenTabParsing() {
        Option<MetricsEvent> resultO = TskvClientMetricsEventLogLine.parse(IOS_OPEN_TAB_EVENT).toMetricsEvent();
        assertTrue(resultO.isPresent());
        MetricsEvent result = resultO.get();
        assertEquals("18895", result.getApiKey());
        assertEquals("iOS", result.getAppPlatform());
        assertEquals("feed", result.getEventName());
        assertEquals(new Instant(1568538292000L), result.getTimestamp());
        assertEquals(613694217, result.getAccountId());
        assertTrue(result.getEventValue().isPresent());
        assertTrue(result.getType() == MetricsEventType.OPEN_LENTA_TAB);
    }

    @Test
    public void testIosUiScrollParsing() {
        Option<MetricsEvent> resultO = TskvClientMetricsEventLogLine.parse(IOS_UI_SCROLL_EVENT).toMetricsEvent();
        assertTrue(resultO.isPresent());
        MetricsEvent result = resultO.get();
        assertEquals("18895", result.getApiKey());
        assertEquals("iOS", result.getAppPlatform());
        assertEquals("feed", result.getEventName());
        assertEquals(new Instant(1568426038000L), result.getTimestamp());
        assertEquals(551130820, result.getAccountId());
        assertTrue(result.getEventValue().isPresent());
        assertTrue(result.getType() == MetricsEventType.OPEN_LENTA_TAB);
    }

    @Test
    public void testAndroidFeedPartitionParsing() {
        Option<MetricsEvent> resultO = TskvClientMetricsEventLogLine.parse(ANDROID_OPEN_TAB_EVENT).toMetricsEvent();
        assertTrue(resultO.isPresent());
        MetricsEvent result = resultO.get();
        assertEquals("18895", result.getApiKey());
        assertEquals("android", result.getAppPlatform());
        assertEquals("fragment_activity", result.getEventName());
        assertEquals(new Instant(1568550022000L), result.getTimestamp());
        assertEquals(803496840, result.getAccountId());
        assertTrue(result.getEventValue().isPresent());
        assertTrue(result.getType() == MetricsEventType.OPEN_LENTA_TAB);
    }

    @Test
    public void testNonLentaOpenTabParsing() {
        Option<MetricsEvent> resultO = TskvClientMetricsEventLogLine.parse(EVENT_CLIENT_NON_LENTA_TAB_OPEN).toMetricsEvent();
        assertTrue(resultO.isPresent());
        assertEquals(MetricsEventType.UNKNOWN, resultO.get().getType());
    }
}
