package ru.yandex.chemodan.eventlog.events.eventlog;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.eventlog.EventLogListenerSupport;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.reflection.ClassX;

/**
 * Abstract event log listener. To implement an event log listener, every event type to be  processed needs to be
 * registered using {@link #registerProcessor(Class, Function2V)} method.
 *
 * @author buberman
 */
public class CallbackEventLogListener extends EventLogListenerSupport {
    private static final Logger logger = LoggerFactory.getLogger(CallbackEventLogListener.class);

    private MapF<Class<? extends AbstractEvent>, Function2V<DataApiUserId, AbstractEvent>> processors = Cf.hashMap();

    /**
     * Register a method to process some event type. This method must use the declared type as a second parameter.
     */
    protected <T extends AbstractEvent> void registerProcessor(Class<T> eventClass, Function2V<DataApiUserId, T> processor) {
        processors.put(eventClass, processor.uncheckedCast());
    }

    protected <T extends AbstractEvent> void registerProcessor(Class<T> eventClass, Function1V<T> processor) {
        registerProcessor(eventClass, (uid, event) -> processor.apply(event));
    }

    @Override
    public void processEvent(AbstractEvent event) {
        Option<DataApiUserId> uidO = toDataApiUid(event.getUid());

        if (uidO.isPresent()) {
            DataApiUserId uid = uidO.get();

            runEventProcessor(uid, event);

        } else {
            incSkippedEvents(event.metadata.tskvEventType);
        }
    }

    protected void runEventProcessor(DataApiUserId uid, AbstractEvent event) {
        ClassX<? extends AbstractEvent> clazz = ClassX.wrap(event.getClass());

        for (; clazz.isAssignableTo(AbstractEvent.class); clazz = clazz.getSuperclass().get().uncheckedCast()) {
            if (processors.containsKeyTs(clazz.getClazz())) {
                processors.getTs(clazz.getClazz()).apply(uid, event);
                return;
            }
        }
        processOtherEvent(uid, event);
    }

    protected Option<DataApiUserId> toDataApiUid(MpfsUid mpfsUid) {
        return mpfsUid.getUidO().map(DataApiPassportUserId::new);
    }

    /**
     * Override to introduce default event processing logic.
     */
    protected void processOtherEvent(DataApiUserId uid, AbstractEvent event) {
        logger.debug("Processor not defined for event type: " + event.getClass().getName());

        incSkippedEvents(event.metadata.tskvEventType);
    }
}
