package ru.yandex.chemodan.eventlog.events.fs;

import java.util.EnumSet;
import java.util.Set;

import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public class FsEvent extends AbstractFsEvent {
    private static final Duration ONE_MS = Duration.millis(1);

    //TODO: it should not be here
    @BenderIgnore
    private static final Set<FsEventType> notFilteredByResourceTypes = EnumSet.of(FsEventType.FS_TRASH_APPEND);

    @BenderIgnore
    private final FsEventType eventType;

    @BenderFlatten
    public final FsEventResourceChange resourceChange;

    @BenderFlatten
    public final FsEventExtra extra;

    @BenderPart(name = "user_uid", strictName = true)
    public final MpfsUid performerUid;

    FsEvent(FsEventType eventType, EventMetadata metadata, FsEventResourceChange resourceChange) {
        this(eventType, metadata, resourceChange, FsEventExtra.NONE);
    }

    FsEvent(FsEventType eventType, EventMetadata metadata, FsEventResourceChange resourceChange, FsEventExtra extra) {
        this(eventType, metadata, resourceChange, extra, metadata.uid);
    }

    FsEvent(FsEventType eventType, EventMetadata metadata, FsEventResourceChange resourceChange, FsEventExtra extra,
            MpfsUid performerUid)
    {
        super(metadata);
        this.eventType = eventType;
        this.resourceChange = resourceChange;
        this.extra = extra;
        this.performerUid = performerUid;
    }

    public FsEvent withoutSource() {
        return new FsEvent(eventType, metadata, resourceChange.withoutSource(), extra, performerUid);
    }

    public FsEvent withoutTarget() {
        return new FsEvent(eventType, metadata, resourceChange.withoutTarget(), extra, performerUid);
    }

    public FsEvent withPerformer(MpfsUid performerUid) {
        return new FsEvent(eventType, metadata, resourceChange, extra, performerUid);
    }

    public static FsEvent parse(TskvEventLogLine tskv) {
        FsEvent event = new FsEvent(
                FsEventType.R.byTskvType(tskv.getEventType()),
                tskv.getMetadata(),
                tskv.getFsResourceChange(),
                tskv.getFsExtra(),
                tskv.getPassportUid("user_uid", "uid"));

        // hack to reorder screenshot publication event and store event
        // https://st.yandex-team.ru/CHEMODAN-31226
        if (event.getEventType() == EventType.FS_STORE) {
            event = event.subtract1msFromCommitTime();
        }

        return event;
    }

    @Override
    public EventType getEventType() {
        return eventType.getGlobalType(this);
    }

    public FsEventType getFsEventType() {
        return eventType;
    }

    @Override
    public ListF<Object> getGroupChunks() {
        return Cf.list()
                .plus1(performerUid)
                .plus(extra.typeSubtype.type)
                .plus(extra.typeSubtype.subtype)
                .plus(resourceChange.source.map(ResourceLocation::getParentAddressPath))
                .plus(resourceChange.target.map(ResourceLocation::getParentAddressPath))
                .plus(resourceChange.resource.compoundType.getTemporaryGroup());
    }

    @Override
    public ListF<Object> getNameChunks() {
        return Cf.list(resourceChange.getName());
    }

    @Override
    protected boolean rejectInternal() {
        return !notFilteredByResourceTypes.contains(eventType) && resourceChange.reject();
    }

    //xxx: why is it here?
    public static FsEvent consForTest(FsEventType type, EventMetadata metadata, FsEventResourceChange change) {
        return new FsEvent(type, metadata, change);
    }

    private FsEvent subtract1msFromCommitTime() {
        return new FsEvent(eventType, metadata.subtractFromCommitTime(ONE_MS), resourceChange, extra, performerUid);
    }
}
