package ru.yandex.chemodan.eventlog.events.fs;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.events.SourceInfo;
import ru.yandex.chemodan.eventlog.events.TargetInfo;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public class FsEventResourceChange extends DefaultObject {
    @BenderIgnore
    public final MpfsUid owner;

    @BenderIgnore
    public final Option<ResourceLocation> source;

    @BenderIgnore
    public final Option<ResourceLocation> target;

    @BenderFlatten
    public final Resource resource;

    @BenderFlatten
    public final SourceInfo sourceInfo;

    @BenderFlatten
    public final TargetInfo targetInfo;

    public FsEventResourceChange(
            MpfsUid owner, Option<ResourceLocation> source, Option<ResourceLocation> target, Resource resource)
    {
        this.owner = owner;
        this.source = source;
        this.target = target;
        this.resource = resource;
        this.sourceInfo = SourceInfo.fromAddress(owner, source.map(ResourceLocation::getAddress));
        this.targetInfo = TargetInfo.fromAddress(owner, target.map(ResourceLocation::getAddress));
    }

    public FsEventResourceChange withoutSource() {
        return new FsEventResourceChange(owner, Option.empty(), target, resource);
    }

    public FsEventResourceChange withoutTarget() {
        return new FsEventResourceChange(owner, source, Option.empty(), resource);
    }

    public static FsEventResourceChange source(MpfsUid owner, ResourceLocation source, Resource resource) {
        return new FsEventResourceChange(owner, Option.of(source), Option.empty(), resource);
    }

    public static FsEventResourceChange target(MpfsUid owner, ResourceLocation target, Resource resource) {
        return new FsEventResourceChange(owner, Option.empty(), Option.of(target), resource);
    }

    public static FsEventResourceChange sourceAndTarget(
            MpfsUid owner, ResourceLocation source, ResourceLocation target, Resource resource)
    {
        return new FsEventResourceChange(owner, Option.of(source), Option.of(target), resource);
    }

    public String getName() {
        return getAnyAddress().getName();
    }

    private MpfsAddress getAnyAddress() {
        if (target.isPresent()) {
            return target.get().address;
        } else if (source.isPresent()) {
            return source.get().address;
        } else {
            throw new IllegalStateException("At least one of source or target MUST be specified");
        }
    }

    public boolean parentEquals() {
        return source.isPresent() && target.isPresent()
                && source.get().getParentAddress().equals(target.get().getParentAddress());
    }

    public boolean sourceIsForeignAndTargetIsNot() {
        return source.isPresent() && !source.get().address.belongsTo(owner) &&
                target.isPresent() && target.get().address.belongsTo(owner);
    }

    public boolean reject() {
        return sourceAndTargetAreEmptyOrForeign() || targetIsEmptyAndResourceIsForeign();
    }

    private boolean targetIsEmptyAndResourceIsForeign() {
        return !target.isPresent() && !owner.equals(resource.owner);
    }

    private boolean sourceAndTargetAreEmptyOrForeign() {
        return isEmptyOrForeign(source) && isEmptyOrForeign(target);
    }

    private boolean isEmptyOrForeign(Option<ResourceLocation> address) {
        return !address.isPresent() || !address.get().address.belongsTo(owner);
    }
}
