package ru.yandex.chemodan.eventlog.events.fs;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class FsEventTest {

    private static final MpfsUid USER_UID = new MpfsUid(1L);
    private static final MpfsUid OWNER_UID = new MpfsUid(2L);

    @Test
    public void testAcceptWhenSourceIsNotForeign() {
        FsEvent event = createSourceTargetEvent(USER_UID,
                MpfsAddress.parseDir("1:/disk/source/dir/"),
                MpfsAddress.parseDir("3:/disk/target/dir/"));
        Assert.isFalse(event.reject());
    }

    @Test
    public void testRejectWhenSourceAndTargetAreForeign() {
        FsEvent event = createSourceTargetEvent(USER_UID,
                MpfsAddress.parseDir("2:/disk/source/dir/"),
                MpfsAddress.parseDir("3:/disk/target/dir/"));
        Assert.isTrue(event.reject());
    }

    @Test
    public void testRejectCopyWhenTargetIsNoneAndDoNotOwnResource() {
        FsEventResourceChange resourceChange = FsEventResourceChange.source(
                USER_UID,
                new ResourceLocation(MpfsAddress.parseFile(USER_UID + ":/disk/file.txt"), Option.empty()),
                Resource.file("text", "123", OWNER_UID)
        );
        FsEvent event = new FsEvent(
                FsEventType.FS_COPY,
                new EventMetadata(USER_UID, Instant.now(), Option.empty()),
                resourceChange
        );
        Assert.isTrue(event.reject());
    }

    private FsEvent createSourceTargetEvent(MpfsUid uid, MpfsAddress source, MpfsAddress target) {
        return new FsEvent(
                FsEventType.FS_COPY,
                new EventMetadata(uid, Instant.now(), Option.empty()),
                FsEventResourceChange.sourceAndTarget(uid,
                        new ResourceLocation(source, Option.empty()),
                        new ResourceLocation(target, Option.empty()),
                        Resource.directory("123", uid)),
                FsEventExtra.force(false));
    }
}
