package ru.yandex.chemodan.eventlog.events.lenta;

import java.util.NoSuchElementException;

import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.lenta.actiontype.BlockCreatePinnedActionEventType;
import ru.yandex.chemodan.eventlog.events.lenta.actiontype.BlockTaskMergedActionEventType;
import ru.yandex.chemodan.eventlog.events.lenta.actiontype.BlockTaskScheduledActionEventType;
import ru.yandex.chemodan.eventlog.events.lenta.actiontype.BlockUpdateAndUpActionEventType;
import ru.yandex.chemodan.eventlog.events.lenta.actiontype.LentaActionEventType;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.chemodan.eventlog.log.TskvEventType;
import ru.yandex.misc.enums.StringEnumResolver;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author bursy
 */
public class LentaLogEventParser {
    private static final Logger logger = LoggerFactory.getLogger(LentaLogEventParser.class);

    /**
     * Parses the tskv line to AbstractEvent based on event_type and action_event properties
     * @param tskv tskv line to parse
     * @return parsed AbstractEvent
     */
    public static AbstractEvent parse(TskvEventLogLine tskv) {
        switch (tskv.getEventType()) {
            case LENTA_BLOCK_CREATE_PINNED:
                return parseByActionEvent(BlockCreatePinnedActionEventType.R, tskv, tskv.getEventType());
            case LENTA_BLOCK_TASK_MERGED:
                return parseByActionEvent(BlockTaskMergedActionEventType.R, tskv, tskv.getEventType());
            case LENTA_BLOCK_TASK_SCHEDULED:
                return parseByActionEvent(BlockTaskScheduledActionEventType.R, tskv, tskv.getEventType());
            case LENTA_BLOCK_UPDATE_AND_UP:
                return parseByActionEvent(BlockUpdateAndUpActionEventType.R, tskv, tskv.getEventType());
            default:
                // event type is not supported yet, no big deal
                return LentaLogDummyEvent.INSTANCE;
        }
    }

    /**
     * Parses the tskv entry based on action_event property
     * @param resolver Resolver for specific LentaActionEventType
     * @param tskv tskv line to parse
     * @param tskvType event type of tskv line
     * @return parsed AbstractEvent
     */
    private static <T extends Enum<T> & LentaActionEventType> AbstractEvent parseByActionEvent(
            StringEnumResolver<T> resolver, TskvEventLogLine tskv, TskvEventType tskvType)
    {
        try {
            String actionEvent = tskv.getString("action_event");
            LentaActionEventType type = resolver.fromValue(actionEvent);

            return type.parseToAbstractEvent(tskv);
        } catch (NoSuchElementException e) {
            // action event type is not supported, no big deal
            return LentaLogDummyEvent.INSTANCE;
        }
    }

}
