package ru.yandex.chemodan.eventlog.events.sharing;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Either;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.EventType;
import ru.yandex.chemodan.eventlog.events.UniverseInvite;
import ru.yandex.chemodan.eventlog.log.TskvEventLogLine;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@Bendable
public class ShareEvent extends AbstractEvent {
    @BenderIgnore
    public final ShareEventType type;

    @BenderFlatten
    public final ShareData data;

    @BenderPart(name = "user_uid", strictName = true)
    public final Option<MpfsUid> user;

    @BenderFlatten
    public final ShareRightsChange rightsChange;

    @BenderFlatten
    public final UniverseInvite invite;

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data) {
        this(type, metadata, data, Option.empty(), ShareRightsChange.NONE, UniverseInvite.EMPTY);
    }

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data, MpfsUid user) {
        this(type, metadata, data, Option.of(user), ShareRightsChange.NONE, UniverseInvite.EMPTY);
    }

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data, MpfsUid user, ShareRightsChange rights) {
        this(type, metadata, data, Option.of(user), rights, UniverseInvite.EMPTY);
    }

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data,
            Either<MpfsUid, UniverseInvite> userOrInvite, ShareRightsChange rightsChange)
    {
        this(type, metadata, data, userOrInvite.leftO(), userOrInvite.rightO().getOrElse(UniverseInvite.EMPTY),
                rightsChange);
    }

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data, Option<MpfsUid> inviteUid,
            UniverseInvite invite, ShareRightsChange rightsChange)
    {
        this(type, metadata, data, inviteUid, rightsChange, invite);
    }

    ShareEvent(ShareEventType type, EventMetadata metadata, ShareData data, Option<MpfsUid> user,
            ShareRightsChange rightsChange, UniverseInvite invite) {
        super(metadata);
        this.type = type;
        this.data = data;
        this.user = (type == ShareEventType.UNSHARE_FOLDER) ? Option.of(metadata.uid) : user;
        this.rightsChange = rightsChange;
        this.invite = invite;

    }

    public static ShareEvent parse(TskvEventLogLine tskv) {
        return new ShareEvent(
                ShareEventType.R.byTskvType(tskv.getEventType()),
                tskv.getMetadata(),
                tskv.getShareData(),
                tskv.getPassportUidO("user_uid"),
                tskv.getShareInvite(),
                tskv.getShareRightsChange());
    }

    @Override
    public EventType getEventType() {
        return type.getGlobalType(this);
    }

    @Override
    public ListF<Object> getGroupChunks() {
        return Cf.list(data.target.path);
    }

    @Override
    public ListF<Object> getNameChunks() {
        return Cf.list().plus(user);
    }

    public boolean isOwnerEvent() {
        return data.currentUserType == ShareUserType.OWNER;
    }

    public boolean isUserEvent() {
        return user.isSome(getUid());
    }

    @BenderPart(name = "user_type", strictName = true)
    @SuppressWarnings("unused")
    public Option<ShareUserType> getAlteredUserType() {
        return type.userType;
    }

    @Override
    protected boolean acceptInternal() {
        return getEventType() == EventType.SHARE_UNSHARE_FOLDER_USER
                || isOwnerEvent()
                || isUserEvent();
    }

    public static ShareEvent consForTest(ShareEventType type, MpfsUid user, EventMetadata metadata, ShareData data) {
        return new ShareEvent(type, metadata, data, user);
    }
}
