package ru.yandex.chemodan.eventlog.events.sharing;

import org.joda.time.Instant;
import org.junit.Test;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class ShareEventTest {
    private static final MpfsUid OWNER_UID = new MpfsUid(1L);

    private static final MpfsUid SUBSCRIBER_UID = new MpfsUid(2L);

    private static final MpfsUid USER_UID = new MpfsUid(10L);

    @Test
    public void acceptActivateInviteForOwner() {
        Assert.assertFalse(createEvent(ShareEventType.ACTIVATE_INVITE, OWNER_UID).reject());
    }

    @Test
    public void acceptActivateInviteForUser() {
        Assert.assertFalse(createEvent(ShareEventType.ACTIVATE_INVITE, USER_UID).reject());
    }

    @Test
    public void rejectActivateInviteForSubscriber() {
        Assert.assertTrue(createEvent(ShareEventType.ACTIVATE_INVITE, SUBSCRIBER_UID).reject());
    }

    @Test
    public void acceptUnshareFolderForOwner() {
        Assert.assertFalse(createEvent(ShareEventType.UNSHARE_FOLDER_USER, OWNER_UID).reject());
    }

    @Test
    public void acceptUnshareFolderForSubscriber() {
        Assert.assertFalse(createEvent(ShareEventType.UNSHARE_FOLDER_USER, SUBSCRIBER_UID).reject());
    }

    private static ShareEvent createEvent(ShareEventType eventType, MpfsUid uid) {
        EventMetadata metadata = new EventMetadata(uid, Instant.now(), Option.empty());
        MpfsAddress fileAddress = MpfsAddress.parseDir(uid + ":/disk/shared_dir/");

        switch (eventType) {
            case ACTIVATE_INVITE:
                return ShareEvents.activateInvite(
                        metadata,
                        new ShareData(metadata, OWNER_UID, "123", fileAddress, false, Option.empty()),
                        USER_UID,
                        ShareRights.READ_ONLY
                );

            case UNSHARE_FOLDER_USER:
                return ShareEvents.unshareFolderUser(
                        metadata,
                        new ShareData(metadata, OWNER_UID, "123", fileAddress, false, Option.empty()),
                        USER_UID
                );

            default:
                throw new IllegalArgumentException("do not know how to create event type = " + eventType);
        }
    }
}
