package ru.yandex.chemodan.eventlog.log;

import java.math.BigInteger;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.eventlog.events.AbstractEvent;
import ru.yandex.chemodan.eventlog.events.EventMetadata;
import ru.yandex.chemodan.eventlog.events.MpfsAddress;
import ru.yandex.chemodan.eventlog.events.MpfsPath;
import ru.yandex.chemodan.eventlog.events.Resource;
import ru.yandex.chemodan.eventlog.events.ResourceLocation;
import ru.yandex.chemodan.eventlog.events.ResourceType;
import ru.yandex.chemodan.eventlog.events.UniverseInvite;
import ru.yandex.chemodan.eventlog.events.YandexCloudRequestId;
import ru.yandex.chemodan.eventlog.events.album.Album;
import ru.yandex.chemodan.eventlog.events.album.AlbumItem;
import ru.yandex.chemodan.eventlog.events.album.AlbumItemEventType;
import ru.yandex.chemodan.eventlog.events.album.AlbumItemType;
import ru.yandex.chemodan.eventlog.events.billing.Payment;
import ru.yandex.chemodan.eventlog.events.billing.PaymentStatus;
import ru.yandex.chemodan.eventlog.events.billing.Product;
import ru.yandex.chemodan.eventlog.events.fs.FsEventExtra;
import ru.yandex.chemodan.eventlog.events.fs.FsEventResourceChange;
import ru.yandex.chemodan.eventlog.events.fs.StoreTypeSubtype;
import ru.yandex.chemodan.eventlog.events.sharing.ShareData;
import ru.yandex.chemodan.eventlog.events.sharing.ShareRights;
import ru.yandex.chemodan.eventlog.events.sharing.ShareRightsChange;
import ru.yandex.chemodan.mpfs.MpfsResourceId;
import ru.yandex.chemodan.mpfs.MpfsUid;
import ru.yandex.misc.lang.DefaultToString;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TskvEventLogLine extends DefaultToString {
    private final TskvLogLine tskv;

    public TskvEventLogLine(TskvLogLine tskv) {
        this.tskv = tskv;
    }

    public static TskvEventLogLine parse(String line) {
        return new TskvEventLogLine(TskvLogLine.parse(line));
    }

    public Option<AbstractEvent> toEvent() {
        return getEventTypeO().map(eventType -> eventType.parseEvent(this));
    }

    public boolean hasKey(String key) {
        return tskv.containsKey(key);
    }

    public String getString(String key) {
        return tskv.getString(key);
    }

    public String getNonEmptyString(String key) {
        return tskv.getNonEmptyString(key);
    }

    public Option<String> getNonEmptyStringO(String key) {
        return tskv.getNonEmptyStringO(key);
    }

    public BigInteger getBigInteger(String key) {
        return tskv.getBigInteger(key);
    }

    public Option<Integer> getIntegerO(String key) {
        return tskv.getIntegerO(key);
    }

    public Option<Boolean> getBooleanO(String key) {
        return tskv.getBooleanO(key);
    }

    public Boolean getBoolean(String key) {
        return tskv.getBoolean(key);
    }

    public MpfsUid getPassportUid(String key) {
        return getPassportUidO(key).get();
    }

    public MpfsUid getPassportUid(String... keys) {
        return Cf.list(keys)
                .map(this::getPassportUidO)
                .filter(Option::isPresent)
                .map(Option<MpfsUid>::get)
                .first();
    }

    public Option<MpfsUid> getPassportUidO(String key) {
        return tskv.getNonEmptyStringO(key).map(MpfsUid::new);
    }

    public MpfsAddress getAddress(String key) {
        return getAddressO(key).get();
    }

    public Option<MpfsAddress> getAddressO(String key) {
        return tskv.getNonEmptyStringO(key).map(addressStr -> MpfsAddress.parse(addressStr, getResourceTypeO()));
    }

    public MpfsResourceId getResourceId() {
        return MpfsResourceId.parse(tskv.getNonEmptyString("resource_id"));
    }

    public Option<MpfsResourceId> getResourceIdO(String key) {
        return tskv.getNonEmptyStringO(key).map(MpfsResourceId::parse);
    }

    public Option<ResourceLocation> getResourceLocationO(String direction) {
        return getAddressO(direction + "_rawaddress")
                .map(addr -> new ResourceLocation(addr, getResourceIdO(direction + "_folder_id")));
    }

    private MpfsAddress getDirectoryAddressByOwnerAndPath(String ownerKey, String pathKey) {
        return new MpfsAddress(
                getPassportUid(ownerKey),
                MpfsPath.parse(getString(pathKey), ResourceType.DIRECTORY)
        );
    }

    public TskvEventType getEventType() {
        return getEventTypeO().get();
    }

    public Option<TskvEventType> getEventTypeO() {
        return TskvEventType.R.fromValueO(getEventTypeStr());
    }

    private String getEventTypeStr() {
        return getEventTypeStrO().get();
    }

    public Option<String> getEventTypeStrO() {
        return getNonEmptyStringO(getEventTypeFieldName());
    }

    public EventMetadata getMetadata() {
        return getMetadata(getPassportUid("uid"));
    }

    public EventMetadata getMetadata(MpfsUid uid) {
        return new EventMetadata(
                getEventType(), uid,
                tskv.getInstantFromSeconds("unixtime"),
                tskv.getNonEmptyStringO("req_id").map(YandexCloudRequestId::parse)
        ).withHost(tskv.getNonEmptyStringO("hostname"));
    }

    public Resource getResource() {
        return getBasicResource()
                .withKeyAndUrlO(getNonEmptyStringO("public_key"), getNonEmptyStringO("short_url"))
                .withOverwrittenO(getBooleanO("overwritten"));
    }

    protected String getEventTypeFieldName() {
        return "event_type";
    }

    private Resource getBasicResource() {
        ResourceType type = getResourceType();
        switch (type) {
            case FILE:
                return Resource.file(
                        getNonEmptyString("resource_media_type"),
                        getNonEmptyString("resource_file_id"),
                        getPassportUid("owner_uid")
                );

            case DIRECTORY:
                return Resource.directory(
                        getNonEmptyString("resource_file_id"),
                        getPassportUid("owner_uid")
                );

            default:
                throw new IllegalArgumentException("Unknown resource type = " + type);
        }
    }

    private ResourceType getResourceType() {
        return getResourceTypeO().get();
    }

    private Option<ResourceType> getResourceTypeO() {
        return getNonEmptyStringO("resource_type").map(ResourceType.R::fromValue);
    }

    public UniverseInvite getUniverseInvite(String loginKey, String serviceKey) {
        return new UniverseInvite(getString(loginKey), getString(serviceKey));
    }

    public FsEventExtra getFsExtra() {
        return new FsEventExtra(getFsStoreTypes(), getBooleanO("force"));
    }

    public StoreTypeSubtype getFsStoreTypes() {
        if (hasKey("type") && hasKey("subtype")) {
            return new StoreTypeSubtype(getString("type"), getString("subtype"));
        } else {
            return StoreTypeSubtype.NONE;
        }
    }

    public FsEventResourceChange getFsResourceChange() {
        return new FsEventResourceChange(
                getPassportUid("uid"),
                getResourceLocationO("src"),
                getResourceLocationO("tgt"),
                getResource());
    }

    public AlbumItemEventType getAlbumItemEventType() {
        switch (getEventType()) {
            case ALBUM_CREATE_ITEM:
                return AlbumItemEventType.CREATE;

            case ALBUM_ITEMS_APPEND:
                return AlbumItemEventType.APPEND;

            case ALBUM_ITEMS_REMOVE:
                return AlbumItemEventType.REMOVE;

            default:
                throw new IllegalArgumentException("Can't handle type = " + getEventType());
        }
    }

    public Album getAlbum() {
        return new Album(getString("album_id"), getString("album_title"));
    }

    public AlbumItem getAlbumItem() {
        return new AlbumItem(
                AlbumItemType.R.fromValue(getString("album_item_type")),
                getString("album_item_id"),
                getAlbum());
    }

    public Payment getSharePayment() {
        return new Payment(getString("currency"), getString("price"), getSharePaymentStatus()
        );
    }

    public Product getProduct() {
        return new Product(
                getString("product_id"),
                getProductNames(),
                getNonEmptyStringO("product_period"),
                getBooleanO("product_is_free").getOrElse(false)
        );
    }

    private MapF<String, String> getProductNames() {
        return tskv.keys()
                .filter(s -> s.matches("product_name_[a-z]{2}"))
                .toMapMappingToValue(this::getString);
    }

    private PaymentStatus getSharePaymentStatus() {
        return PaymentStatus.cons(tskv.getNonEmptyStringO("status"), tskv.getNonEmptyStringO("status_code"));
    }

    public ShareData getShareData() {
        return new ShareData(
                getMetadata(),
                getPassportUid("owner_uid"),
                getString("gid"),
                getDirectoryAddressByOwnerAndPath("uid", "path"),
                getBoolean("is_invite"),
                getNonEmptyStringO("invite_hash"));
    }

    public UniverseInvite getShareInvite() {
        if (hasKey("user_universe_login")) {
            return getUniverseInvite("user_universe_login", "user_universe_service");
        } else {
            return UniverseInvite.EMPTY;
        }
    }

    public ShareRightsChange getShareRightsChange() {
        return ShareRightsChange.cons(getShareRightsO("rights"), getShareRightsO("prev_rights"));
    }

    private Option<ShareRights> getShareRightsO(String key) {
        return tskv.getIntegerO(key).map(ShareRights.R::fromValue);
    }
}
