package ru.yandex.chemodan.eventlog.log;

import java.math.BigInteger;

import org.joda.time.DateTimeConstants;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.util.tskv.TskvUtils;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class TskvLogLine {
    private static final SetF<String> EMPTY_VALUES = Cf.set("", "-", "None");

    private static final SetF<String> TRUE_VALUES = Cf.set("1", "true");

    private static final SetF<String> FALSE_VALUES = Cf.set("0", "false");

    private final MapF<String, String> keyValue;

    public TskvLogLine(MapF<String, String> keyValue) {
        this.keyValue = keyValue;
    }

    public static TskvLogLine parse(String line) {
        return new TskvLogLine(TskvUtils.extractTskv(line));
    }

    public String getString(String key) {
        return getStringO(key).getOrThrow("log doesn't contain expected key = " + key + ", all values =" + keyValue);
    }

    public Option<String> getStringO(String key) {
        return keyValue.getO(key);
    }

    public String getNonEmptyString(String key) {
        Option<String> nonEmptyStringO = getNonEmptyStringO(key);
        return nonEmptyStringO.getOrThrow("log doesn't contain expected key = " + key + ", all values =" + keyValue);
    }

    public Option<String> getNonEmptyStringO(String key) {
        return getStringO(key)
                .filter(TskvLogLine::isNotEmpty);
    }

    private static boolean isNotEmpty(String value) {
        return !EMPTY_VALUES.containsTs(value.trim());
    }

    @SuppressWarnings("unused")
    public int getInt(String key) {
        return Integer.parseInt(getNonEmptyString(key));
    }

    public BigInteger getBigInteger(String key) {
        return new BigInteger(getNonEmptyString(key));
    }

    public boolean getBoolean(String key) {
        return getBooleanO(key).get();
    }

    public Option<Boolean> getBooleanO(String key) {
        return getNonEmptyStringO(key).map(TskvLogLine::parseBoolean);
    }

    private static boolean parseBoolean(String value) {
        value = value.toLowerCase();
        if (TRUE_VALUES.containsTs(value)) {
            return true;
        } else if (FALSE_VALUES.containsTs(value)) {
            return false;
        } else {
            throw new IllegalArgumentException("unknown boolean value = " + value);
        }
    }

    public Option<Integer> getIntegerO(String key) {
        return getNonEmptyStringO(key).map(Integer::parseInt);
    }

    public long getLong(String key) {
        return Long.parseLong(getNonEmptyString(key));
    }

    public Instant getInstantFromSeconds(@SuppressWarnings("SameParameterValue") String key) {
        return new Instant(getLong(key) * DateTimeConstants.MILLIS_PER_SECOND);
    }

    public boolean containsKey(String key) {
        return keyValue.containsKeyTs(key);
    }

    public ListF<String> keys() {
        return keyValue.keys();
    }

    public MapF<String, String> toMap() {
        return keyValue;
    }
}
