package ru.yandex.chemodan.grpc.server;

import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.Executors;

import com.google.common.collect.Lists;
import io.grpc.BindableService;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.util.TransmitStatusRuntimeExceptionInterceptor;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.chemodan.grpc.server.interceptors.ContextGrpcServerInterceptor;
import ru.yandex.chemodan.grpc.server.interceptors.LoggingGrpcServerInterceptor;
import ru.yandex.chemodan.grpc.server.interceptors.Tvm2GrpcServerInterceptor;
import ru.yandex.chemodan.grpc.server.services.HealthService;
import ru.yandex.inside.passport.tvm2.Tvm2;
import ru.yandex.inside.passport.tvm2.web.Tvm2BaseContextConfiguration;
import ru.yandex.inside.passport.tvm2.web.Tvm2CheckingMode;
import ru.yandex.misc.thread.factory.ThreadNameIndexThreadFactory;
import ru.yandex.misc.version.AppName;


@Configuration
@Import({Tvm2BaseContextConfiguration.class})
public class GrpcServerConfiguration {

    @Bean
    public Server grpcServer(@Value("${grpc.port:-26118}") int port,
            @Value("${grpc.thread-count:-10}") int threadsCount,
            Optional<List<BindableService>> services,
            Tvm2GrpcServerInterceptor tvm2GrpcServerInterceptor, AppName appName) throws IOException
    {
        ServerBuilder serverBuilder = ServerBuilder.forPort(port);
        for (BindableService service : services.orElse(Lists.newArrayList())) {
            serverBuilder = serverBuilder.addService(service);
        }
        Server server = serverBuilder.addService(new HealthService())
                .intercept(tvm2GrpcServerInterceptor)
                .intercept(TransmitStatusRuntimeExceptionInterceptor.instance())
                .intercept(new LoggingGrpcServerInterceptor())
                .intercept(new ContextGrpcServerInterceptor(appName))
                .executor(Executors
                        .newFixedThreadPool(threadsCount, new ThreadNameIndexThreadFactory("gRPC-server-thread")))
                .build();
        server.start();
        return server;
    }

    @Bean
    public Tvm2GrpcServerInterceptor tvm2GrpcServerInterceptor(Tvm2 tvm2,
            @Value("${tvm.filter.mode:-DISABLED}") Tvm2CheckingMode tvm2Mode)
    {
        return new Tvm2GrpcServerInterceptor(tvm2, tvm2Mode);
    }
}
