package ru.yandex.chemodan.grpc.server.interceptors;

import java.util.Set;
import java.util.concurrent.TimeUnit;

import com.google.common.base.Stopwatch;
import io.grpc.Context;
import io.grpc.ForwardingServerCall;
import io.grpc.Metadata;
import io.grpc.ServerCall;
import io.grpc.ServerCallHandler;
import io.grpc.ServerInterceptor;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.grpc.GrpcContextCommonKeys;
import ru.yandex.inside.passport.tvm2.TvmHeaders;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.web.servlet.HttpRequestUtils;

public class LoggingGrpcServerInterceptor implements ServerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(LoggingGrpcServerInterceptor.class);

    private static final String AUTHORIZATION = "Authorization";
    private static final String YANDEX_AUTHORIZATION = "Yandex-Authorization";
    private static final Set<String> PRIVATE_HEADERS = Cf.set(AUTHORIZATION, YANDEX_AUTHORIZATION);

    private static final SetF<String> COOKIES = Cf.set("Cookie", "Cookie2", "Ya-Client-Cookie");

    @Override
    public <ReqT, RespT> ServerCall.Listener<ReqT> interceptCall(ServerCall<ReqT, RespT> call, Metadata headers,
            ServerCallHandler<ReqT, RespT> next)
    {
        Stopwatch stopwatch = Stopwatch.createStarted();
        ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT> forwardingServerCall =
                new ForwardingServerCall.SimpleForwardingServerCall<ReqT, RespT>(call) {
            @Override
            public void sendMessage(RespT message) {
                super.sendMessage(message);
                if (stopwatch.isRunning()) {
                    stopwatch.stop();
                }
                logIncomingCall(call, headers, Option.empty(), stopwatch);
            }
        };
        try {
            return next.startCall(forwardingServerCall, headers);
        } catch (StatusRuntimeException e) {
            if (stopwatch.isRunning()) {
                stopwatch.stop();
            }
            logIncomingCall(call, headers, Option.of(e.getStatus()), stopwatch);
            throw e;
        }
    }

    private <ReqT, RespT> void logIncomingCall(ServerCall<ReqT, RespT> call, Metadata metadata,
            Option<Status> status, Stopwatch stopwatch)
    {
        Context context = Context.current();
        logger.info(
                "request_type=gRPC gRPC_method={} metadata_headers={} rid={} response_time_ms={} status={} ycrid={}",
                call.getMethodDescriptor().getFullMethodName(), getHeaders(metadata),
                GrpcContextCommonKeys.REQUEST_ID.get(context),
                stopwatch.elapsed(TimeUnit.MILLISECONDS), status.map(Status::getCode).getOrElse(Status.Code.OK),
                GrpcContextCommonKeys.YCRID.get(context));
    }

    public static MapF<String, Object> getHeaders(Metadata headers) {
        return Cf.x(headers.keys()).toMapMappingToValue(key -> getMetadataValue(key, headers));
    }

    private static String getMetadataValue(String key, Metadata metadata) {
        if (PRIVATE_HEADERS.contains(key)) {
            return HttpRequestUtils.KnownAuthPrefix.encodeValue(getMetadataValue(key, metadata));
        } else if (TvmHeaders.SERVICE_TICKET.equals(key) || TvmHeaders.USER_TICKET.equals(key)) {
            return HttpRequestUtils.removeTvmTicketSignature(getMetadataValueOrEmpty(key, metadata));
        } else if (COOKIES.containsTs(key)) {
            return HttpRequestUtils.removeSessionSecrets(getMetadataValueOrEmpty(key, metadata));
        }
        return getMetadataValueOrEmpty(key, metadata);
    }

    private static String getMetadataValueOrEmpty(String key, Metadata metadata) {
        return Option.ofNullable(metadata.get(Metadata.Key.of(key, Metadata.ASCII_STRING_MARSHALLER))).getOrElse("");
    }

}
