package ru.yandex.chemodan.app.lentaloader.cool.utils;

import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.Check;
import ru.yandex.misc.random.Random2;

/**
 * @author tolmalev
 */
public class BlockTitlesGenerator {
    public static final ListF<Language> SUPPORTED_LANGUAGES = Cf.list(Language.RUSSIAN, Language.ENGLISH, Language.UKRAINIAN, Language.TURKISH);

    private final MapF<IntervalType, OneIntervalGenerator> generators;

    private final TextProcessor common;
    private final GeoNamesSource geoNamesSource;

    public BlockTitlesGenerator(GeoNamesSource geoNamesSource) {
        this(true, geoNamesSource);
    }

    public BlockTitlesGenerator(boolean validateTemplates, GeoNamesSource geoNamesSource) {
        this.geoNamesSource = geoNamesSource;
        this.common = new TextProcessor(geoNamesSource);
        this.generators = Cf.<OneIntervalGenerator>list(
                new OneDayGenerator(geoNamesSource),
                new WeekendGenerator(geoNamesSource),
                new WeekGenerator(geoNamesSource),
                new MonthGenerator(geoNamesSource),
                new SeasonGenerator(geoNamesSource),
                new YearGenerator(geoNamesSource)
        ).toMapMappingToKey(OneIntervalGenerator::supportedType);

        // Валидируем все шаблоны на момент старта приложения
        if (validateTemplates) {
            validateAllTemplates(2);
        }
    }

    public String processSimplePlaceholder(String placeholder, Language language, TitleGenerationContext context) {
        return common.processSimpleTemplate(placeholder, language, context);
    }

    public String processTemplate(String placeholder, Language language, TitleGenerationContext context) {
        return common.processTemplate(placeholder, language, context);
    }

    public CommonTextGenerator createUniversalGenerator(String filename, Class clazz) {
        return new CommonTextGenerator(new ClassPathResourceInputStreamSource(clazz, filename), geoNamesSource);
    }

    public boolean isSupportedPlaceholder(String placeholder) {
        return common.isSupportedSimpleTemplate(placeholder);
    }

    public LocalizedString generateBlockTitle(TitleGenerationContext context) {
        OneIntervalGenerator generator = generators.getOrThrow(context.intervalType, "Can't find generator for interval " + context.intervalType);
        return generator.generateTitle(context);
    }

    public LocalizedString generateBlockSubtitle(TitleGenerationContext context) {
        OneIntervalGenerator generator = generators.getOrThrow(context.intervalType, "Can't find generator for interval " + context.intervalType);
        return generator.generateSubtitle(context);
    }

    public LocalizedString generatePhotosliceLinkText(TitleGenerationContext context) {
        OneIntervalGenerator generator = generators.getOrThrow(context.intervalType, "Can't find generator for interval " + context.intervalType);
        return generator.generatePhotosliceLinkText(context);
    }

    public LocalizedString generateLegacyBlockTitle(TitleGenerationContext context) {
        OneIntervalGenerator generator = generators.getOrThrow(context.intervalType, "Can't find generator for interval " + context.intervalType);
        return generator.generateLegacyBlockTitle(context);
    }

    void validateAllTemplates(int iterationsPerDate) {
        generators.values().forEach(generator -> {
            IntervalType intervalType = generator.supportedType();

            DateTime time = DateTime.parse("2013-01-01T00:00:00.00+0300");
            DateTime wholeEnd = DateTime.parse("2020-01-01T00:00:00.00+0300");

            while (time.isBefore(wholeEnd)) {
                if (!intervalType.getTypesForDateTime(time).containsTs(intervalType)) {
                    time = time.plusDays(1);
                    continue;
                }

                DateTime start = intervalType.getIntervalStart(time);
                DateTime end = intervalType.getIntervalStart(time);

                Cf.range(0, iterationsPerDate).forEach((i) -> {
                    SUPPORTED_LANGUAGES.forEach(lang -> {
                        try {
                            TitleGenerationContext context = new TitleGenerationContext(Random2.R, intervalType, start);

                            generator.generateTitle(context).get(lang);
                            generator.generateSubtitle(context).get(lang);
                            generator.generatePhotosliceLinkText(context).get(lang);
                        } catch (RuntimeException e) {
                            e.printStackTrace();
                            throw e;
                        }
                    });
                });

                time = time.plusDays(1);
            }
        });
    }

    public boolean isSupportedLanguage(Language lang) {
        return SUPPORTED_LANGUAGES.containsTs(lang);
    }
}

interface OneIntervalGenerator {
    LocalizedString generateTitle(TitleGenerationContext context);
    LocalizedString generateSubtitle(TitleGenerationContext context);
    LocalizedString generatePhotosliceLinkText(TitleGenerationContext context);
    LocalizedString generateLegacyBlockTitle(TitleGenerationContext context);

    IntervalType supportedType();
}

abstract class GeneratorBase implements OneIntervalGenerator {
    protected CommonTextGenerator universal;

    public GeneratorBase(String modelFilename, GeoNamesSource geoNamesSource) {
        this.universal = new CommonTextGenerator(new ClassPathResourceInputStreamSource(GeneratorBase.class, modelFilename), geoNamesSource);
    }

    protected LocalizedString processTemplateByName(String templateName, TitleGenerationContext context) {
        return universal.processNamedTemplates(templateName, context);
    }
}

abstract class SimpleGenerator extends GeneratorBase {
    private final IntervalType intervalType;

    public SimpleGenerator(String modelFilename, IntervalType intervalType, GeoNamesSource geoNamesSource) {
        this(modelFilename, intervalType, geoNamesSource, Cf.list("title_variants", "subtitle_variants", "photoslice_link_text", "legacy_block_title"));
    }

    public SimpleGenerator(String modelFilename, IntervalType intervalType, GeoNamesSource geoNamesSource, ListF<String> requiredTemplates) {
        super(modelFilename, geoNamesSource);
        this.intervalType = intervalType;

        requiredTemplates.forEach(template -> Check.isTrue(universal.hasTemplate(template)));
    }

    @Override
    public LocalizedString generateTitle(TitleGenerationContext context) {
        return processTemplateByName("title_variants", context);
    }

    @Override
    public LocalizedString generateSubtitle(TitleGenerationContext context) {
        return processTemplateByName("subtitle_variants", context);
    }

    @Override
    public LocalizedString generatePhotosliceLinkText(TitleGenerationContext context) {
        return processTemplateByName("photoslice_link_text", context);
    }

    @Override
    public LocalizedString generateLegacyBlockTitle(TitleGenerationContext context) {
        return processTemplateByName("legacy_block_title", context);
    }

    @Override
    public IntervalType supportedType() {
        return intervalType;
    }
}

class OneDayGenerator extends SimpleGenerator {
    public OneDayGenerator(GeoNamesSource geoNamesSource) {
        super("one_day.json", IntervalType.ONE_DAY, geoNamesSource);
    }
}

class WeekGenerator extends SimpleGenerator {
    public WeekGenerator(GeoNamesSource geoNamesSource) {
        super("week.json", IntervalType.WEEK, geoNamesSource);
    }
}

class YearGenerator extends SimpleGenerator {
    public YearGenerator(GeoNamesSource geoNamesSource) {
        super("year.json", IntervalType.YEAR, geoNamesSource);
    }
}

class MonthGenerator extends SimpleGenerator {
    public MonthGenerator(GeoNamesSource geoNamesSource) {
        super("month.json", IntervalType.MONTH, geoNamesSource, Cf.list("title_variants", "subtitle_variants",
                "photoslice_link_text_current_year", "photoslice_link_text_not_current_year", "legacy_block_title"));
    }

    @Override
    public LocalizedString generatePhotosliceLinkText(TitleGenerationContext context) {
        DateTime now = DateTime.now(context.start.getZone());

        if (context.start.getYear() == now.getYear()) {
            return processTemplateByName("photoslice_link_text_current_year", context);
        } else {
            return processTemplateByName("photoslice_link_text_not_current_year", context);
        }
    }
}

class WeekendGenerator extends SimpleGenerator {
    public WeekendGenerator(GeoNamesSource geoNamesSource) {
        super("weekend.json", IntervalType.WEEKEND, geoNamesSource, Cf.list("title_variants", "subtitle_variants_1", "subtitle_variants_2",
                "photoslice_link_text", "legacy_block_title_1", "legacy_block_title_2"));
    }

    @Override
    public LocalizedString generateSubtitle(TitleGenerationContext context) {
        if (!CoolTitlesUtils.isCrossMonth(context.start, IntervalType.WEEKEND)) {
            return processTemplateByName("subtitle_variants_1", context);
        } else {
            return processTemplateByName("subtitle_variants_2", context);
        }
    }

    @Override
    public LocalizedString generateLegacyBlockTitle(TitleGenerationContext context) {
        if (!CoolTitlesUtils.isCrossMonth(context.start, IntervalType.WEEKEND)) {
            return processTemplateByName("legacy_block_title_1", context);
        } else {
            return processTemplateByName("legacy_block_title_2", context);
        }
    }
}

class SeasonGenerator extends SimpleGenerator {
    public SeasonGenerator(GeoNamesSource geoNamesSource) {
        super("season.json", IntervalType.SEASON, geoNamesSource, Cf.list("title_variants", "subtitle_variants", "winter_subtitle_variants",
                "photoslice_link_text", "legacy_block_title", "winter_legacy_block_title"));
    }

    @Override
    public LocalizedString generateSubtitle(TitleGenerationContext context) {
        if (CoolTitlesUtils.isCrossYear(context.start, IntervalType.SEASON)) {
            return processTemplateByName("winter_subtitle_variants", context);
        } else {
            return processTemplateByName("subtitle_variants", context);
        }
    }

    @Override
    public LocalizedString generateLegacyBlockTitle(TitleGenerationContext context) {
        if (CoolTitlesUtils.isCrossYear(context.start, IntervalType.SEASON)) {
            return processTemplateByName("winter_legacy_block_title", context);
        } else {
            return processTemplateByName("legacy_block_title", context);
        }
    }
}
