package ru.yandex.chemodan.app.lentaloader.cool.utils;

import com.fasterxml.jackson.databind.JsonNode;
import org.joda.time.DateTime;
import org.joda.time.DateTimeUtils;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class BlockTitlesGeneratorTest {
    private final GeoNamesSource geoNamesSource = createGeoNamesSource();

    private final BlockTitlesGenerator blockTitlesGenerator = new BlockTitlesGenerator(false, geoNamesSource);

    public static GeoNamesSource createGeoNamesSource() {
        MapF<Integer, MapF<Language, MapF<GeoNamesSource.Case, String>>> map = Cf.hashMap();

        JsonNode node =
                JsonNodeUtils.getNode(new ClassPathResourceInputStreamSource(BlockTitlesGenerator.class, "linguistics_for_region.js").readBytes());

        for(String field: Cf.x(node.fieldNames()).toList()) {
            Integer regionId = Integer.valueOf(field);
            MapF<Language, MapF<GeoNamesSource.Case, String>> regionNames = Cf.hashMap();

            JsonNode rNode = node.get(field);
            for(String langStr: Cf.x(rNode.fieldNames()).toList()) {
                Language language = Language.R.fromValue(langStr);
                MapF<GeoNamesSource.Case, String> byCase = Cf.hashMap();

                JsonNode cNode = rNode.get(langStr);
                for(String caseStr: Cf.x(cNode.fieldNames()).toList()) {
                    byCase.put(GeoNamesSource.Case.R.valueOf(StringUtils.removeEnd(caseStr, "_case")), cNode.get(caseStr).asText());
                }

                regionNames.put(language, byCase);
            }

            map.put(regionId, regionNames);
        }

        return new GeoNamesSource() {

            @Override
            public String getGeoName(int regionId, Case c, Language language) {
                return map.getTs(regionId).getTs(language).getTs(c);
            }
        };
    }

    @Test
    public void processSimplePlaceholderDate() {
        Assert.equals(
                "2019",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.start:yyyy", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-02-17T10:00:00.00+0300")))
        );
        Assert.equals(
                "2020",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.end:yyyy", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-02-17T10:00:00.00+0300")))
        );

        Assert.equals(
                "17",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.start:d", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-02-19T10:00:00.00+0300")))
        );
        Assert.equals(
                "19",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.end:d", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-02-19T10:00:00.00+0300")))
        );

        Assert.equals(
                "1",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.start:d", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "2",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.end:d", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "17.02.2019",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.start:dd.MM.yyyy", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-03-18T10:00:00.00+0300")))
        );
        Assert.equals(
                "18.03.2020",
                blockTitlesGenerator.processSimplePlaceholder(
                        "date.end:dd.MM.yyyy", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-17T10:00:00.00+0300"), DateTime.parse("2020-03-18T10:00:00.00+0300")))
        );
    }

    @Test
    public void processSimplePlaceholderMonth() {
        Assert.equals(
                "январь",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "января",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:gen", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "февраль",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "февраля",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:gen", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-02-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "декабрь",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-12-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "декабря",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.start:gen", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-12-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );


        Assert.equals(
                "декабрь",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.end:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-12-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "март",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.end:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-03-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "февраль",
                blockTitlesGenerator.processSimplePlaceholder(
                        "month.end:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-12-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
    }

    @Test
    public void processSimplePlaceholderSeason() {
        Assert.equals(
                "зима",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.start:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "зимняя",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.start:adj_fem", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "весна",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.end:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-03-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "весенняя",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.end:adj_fem", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-03-2T10:00:00.00+0300")))
        );

        Assert.equals(
                "лето",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.start:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-07-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "летняя",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.start:adj_fem", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-07-1T10:00:00.00+0300"), DateTime.parse("2020-02-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "осень",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.end:nom", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-09-2T10:00:00.00+0300")))
        );
        Assert.equals(
                "осенняя",
                blockTitlesGenerator.processSimplePlaceholder(
                        "season.end:adj_fem", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2020-10-2T10:00:00.00+0300")))
        );
    }

    @Test
    public void capitalizeWorks() {
        Assert.equals(
                "Осенняя подборка",
                blockTitlesGenerator.processTemplate(
                        "{{season.start:adj_fem}} подборка", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-09-1T10:00:00.00+0300"), DateTime.parse("2019-10-2T10:00:00.00+0300")))
                );

        Assert.equals(
                "Подборка осень",
                blockTitlesGenerator.processTemplate(
                        "Подборка {{season.start:nom}}", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-09-1T10:00:00.00+0300"), DateTime.parse("2019-10-2T10:00:00.00+0300")))
                );

        Assert.equals(
                "Вижу зиму",
                blockTitlesGenerator.processTemplate(
                        "Вижу {{season.start:acc}}", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, DateTime.parse("2019-01-1T10:00:00.00+0300"), DateTime.parse("2019-02-2T10:00:00.00+0300")))
                );
    }

    @Test
    public void seasonsEng() {
        Random2 R = new Random2(123);

        Assert.equals(
                "Fall selection",
                blockTitlesGenerator.generateBlockTitle(new TitleGenerationContext(R, IntervalType.SEASON, DateTime.parse("2019-09-01T00:00:00.00+0300"))).get(Language.ENGLISH));
        Assert.equals(
                "Осенняя подборка",
                blockTitlesGenerator.generateBlockTitle(new TitleGenerationContext(R, IntervalType.SEASON, DateTime.parse("2019-09-01T00:00:00.00+0300"))).get(Language.RUSSIAN));
    }

    @Test
    public void fixedVariants() {
        Random2 R = new Random2(123);
        Assert.equals(
                "Яркий день",
                blockTitlesGenerator.generateBlockTitle(new TitleGenerationContext(R, IntervalType.ONE_DAY, DateTime.parse("2019-10-11T00:00:00.00+0300"))).get(Language.RUSSIAN));
        Assert.equals(
                "11 октября 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.ONE_DAY, DateTime.parse("2019-10-11T00:00:00.00+0300"))).get(Language.RUSSIAN));
        Assert.equals(
                "October 11, 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.ONE_DAY, DateTime.parse("2019-10-11T00:00:00.00+0300"))).get(Language.ENGLISH));

        Assert.equals(
                "October 12–13, 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.WEEKEND, DateTime.parse("2019-10-12T00:00:00.00+0300"))).get(Language.ENGLISH));
        Assert.equals(
                "October 12–13, 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.WEEKEND, DateTime.parse("2019-10-12T00:00:00.00+0000"))).get(Language.ENGLISH));
        Assert.equals(
                "October 12–13, 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.WEEKEND, DateTime.parse("2019-10-12T00:00:00.00+1200"))).get(Language.ENGLISH));
        Assert.equals(
                "October 12–13, 2019",
                blockTitlesGenerator.generateBlockSubtitle(new TitleGenerationContext(R, IntervalType.WEEKEND, DateTime.parse("2019-10-12T00:00:00.00+2300"))).get(Language.ENGLISH));
    }

    @Test
    public void currentYearCheck() {
        Random2 R = new Random2(123);
        DateTime now = DateTime.parse("2018-10-12T00:00:00.00+2300");
        DateTimeUtils.setCurrentMillisFixed(now.getMillis());

        Assert.equals(
                "Все фото за июль",
                blockTitlesGenerator.generatePhotosliceLinkText(new TitleGenerationContext(R, IntervalType.MONTH, DateTime.parse("2018-07-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );

        Assert.equals(
                "Все фото за декабрь",
                blockTitlesGenerator.generatePhotosliceLinkText(new TitleGenerationContext(R, IntervalType.MONTH, DateTime.parse("2018-12-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );

        Assert.equals(
                "Все фото за январь",
                blockTitlesGenerator.generatePhotosliceLinkText(new TitleGenerationContext(R, IntervalType.MONTH, DateTime.parse("2018-01-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );

        Assert.equals(
                "Все фото за июль 2017",
                blockTitlesGenerator.generatePhotosliceLinkText(new TitleGenerationContext(R, IntervalType.MONTH, DateTime.parse("2017-07-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );

        DateTimeUtils.setCurrentMillisSystem();
    }

    @Test
    public void legacyBlockTitle() {
        Random2 R = new Random2(123);

        Assert.equals(
                "Осень 2018",
                blockTitlesGenerator.generateLegacyBlockTitle(new TitleGenerationContext(R, IntervalType.SEASON, DateTime.parse("2018-09-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );

        Assert.equals(
                "Сентябрь 2018",
                blockTitlesGenerator.generateLegacyBlockTitle(new TitleGenerationContext(R, IntervalType.MONTH, DateTime.parse("2018-09-01T00:00:00.00+0300"))).get(Language.RUSSIAN)
        );
    }

    @Test
    public void geoNames() {
        Assert.equals(
                "Я еду в Москву",
                blockTitlesGenerator.processTemplate("Я еду {{geo:preposition}} {{geo:accusative}}", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, IntervalType.WEEK, DateTime.now(), Cf.list(213))
                )
        );

        Assert.equals(
                "Я еду из Москвы",
                blockTitlesGenerator.processTemplate("Я еду из {{geo:genitive}}", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, IntervalType.WEEK, DateTime.now(), Cf.list(213))
                )
        );

        Assert.equals(
                "Я еду из Москвы в Украину",
                blockTitlesGenerator.processTemplate("Я еду из {{geo:0.genitive}} {{geo:1.preposition}} {{geo:1.accusative}}", Language.RUSSIAN,
                        new TitleGenerationContext(Random2.R, IntervalType.WEEK, DateTime.now(), Cf.list(213, 187))
                )
        );
    }

    @Test
    public void validateAllTemplates() {
        new OneDayGenerator(GeoNamesSource.EMPTY);
        new WeekendGenerator(GeoNamesSource.EMPTY);
        new WeekGenerator(GeoNamesSource.EMPTY);
        new MonthGenerator(GeoNamesSource.EMPTY);
        new SeasonGenerator(GeoNamesSource.EMPTY);
        new YearGenerator(GeoNamesSource.EMPTY);

        blockTitlesGenerator.validateAllTemplates(100);
    }
}
