package ru.yandex.chemodan.app.lentaloader.cool.utils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.io.InputStreamSource;
import ru.yandex.misc.lang.Check;

public class CommonTextGenerator {

    private final TextProcessor textProcessor;
    private MapF<String, MapF<Language, ListF<String>>> data;

    public CommonTextGenerator(MapF<String, MapF<Language, ListF<String>>> data, GeoNamesSource geoNamesSource) {
        this.data = data;
        this.textProcessor = new TextProcessor(geoNamesSource);
    }

    public CommonTextGenerator(InputStreamSource config, GeoNamesSource geoNamesSource) {
        this(parseInternalFormat(config), geoNamesSource);
    }

    public static MapF<String, MapF<Language, ListF<String>>> parseInternalFormat(InputStreamSource config) {
        MapF<String, MapF<Language, ListF<String>>> data = Cf.hashMap();

        JsonNode node = config.read(inputStream -> new ObjectMapper().reader().readTree(inputStream));

        for (String fieldName : Cf.x(node.fieldNames()).toList()) {
            JsonNode templateNode = node.get(fieldName);
            Check.isTrue(templateNode.isObject(), "bad format for node " + fieldName);

            data.put(fieldName, parseOneTemplate(templateNode));
        }

        return data;
    }

    private static MapF<Language, ListF<String>> parseOneTemplate(JsonNode node) {
        Check.isTrue(node.isObject(), "bad format for node");
        MapF<Language, ListF<String>> result = Cf.hashMap();

        for (Language language : Language.values()) {
            JsonNode langNode = node.get(language.value());
            Check.notNull(langNode, "No localization for lang=" + language);

            if (langNode.isArray()) {
                ArrayNode array = (ArrayNode) langNode;
                result.put(language, Cf.x(array.elements()).map(JsonNode::asText).toList());
            } else if (langNode.isTextual()) {
                result.put(language, Cf.list(langNode.asText()));
            } else {
                Check.fail("Bad node for lang=" + language + ", " + langNode);
            }
        }

        return result;
    }

    public LocalizedString processNamedTemplates(String templateName, TitleGenerationContext context) {
        MapF<Language, ListF<String>> byLang = data.getOrThrow(templateName, () -> "Unknown template name " + templateName);
        ListF<Language> languages = BlockTitlesGenerator.SUPPORTED_LANGUAGES;
        return new DynamicLocalizedString(languages.toMap(lang -> Tuple2.tuple(lang,
                textProcessor.processTemplate(context.R.randomElement(byLang.getTs(lang)), lang, context))));
    }

    public boolean hasTemplate(String templateName) {
        return data.containsKeyTs(templateName);
    }

    @Data
    @BenderBindAllFields
    static class CommonModel {
        @BenderPart(name = "months")
        final ListF<MapF<String, MapF<String, String>>> monthsByLangByCase;
        @BenderPart(name = "seasons")
        final ListF<MapF<String, MapF<String, String>>> seasonsByLangByCase;
    }
}
