package ru.yandex.chemodan.app.lentaloader.cool.utils;

import java.util.Map;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.inside.utils.DynamicLocalizedString;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.io.InputStreamSource;

public class TankerTextGenerator {

    private static final SetF<Integer> RUSSIAN_SECOND_FORM_MODS_VALUES = Cf.set(2, 3, 4);

    private static final SetF<Integer> RUSSIAN_NON_SECOND_FORM_MODS_VALUES = Cf.set(12, 13, 14, 15);

    private final MapF<String, MapF<Language, FormSelector>> tankerConfiguration;
    private final TextProcessor textProcessor;

    public static MapF<String, MapF<Language, FormSelector>> parseTankerFormat(InputStreamSource config) {
        MapF<String, MapF<Language, FormSelector>> data = Cf.hashMap();

        JsonNode node = config.read(inputStream -> new ObjectMapper().reader().readTree(inputStream));

        node = node.get("keysets");

        for (Map.Entry<String, JsonNode> t2 : Cf.x(node.fields()).toList()) {
            JsonNode keysNode = t2.getValue().get("keys");
            String keysetName = t2.getKey();
            for (Map.Entry<String, JsonNode> e2 : Cf.x(keysNode.fields()).toList()) {
                String key = e2.getKey();
                JsonNode translationsNode = e2.getValue().get("translations");
                MapF<Language, FormSelector> byLang = Cf.hashMap();
                for (Language language : Language.values()) {
                    JsonNode langNode = translationsNode.get(language.value());
                    if (langNode == null) {
                        continue;
                    }
                    BiFunction<Object, FormsData, String> selector = (value, formsData) -> formsData.getForms().first();
                    ListF<String> forms = Cf.list(langNode.get("form"), langNode.get("form1"), langNode.get("form2"),
                            langNode.get("form3"), langNode.get("form4")).filter(Objects::nonNull).map(JsonNode::asText);
                    Option<String> numberParameterName0 = TextProcessor.findNumberParameterInForms(forms);
                    if (forms.size() < 2 || !numberParameterName0.isPresent()) {
                        byLang.put(language, new FormSelector(new FormsData(forms, ""), selector));
                        continue;
                    }
                    if (language == Language.RUSSIAN || language == Language.UKRAINIAN) {
                        selector = TankerTextGenerator::selectRussianAndUkrainianNumberForm;
                    } else if (language == Language.ENGLISH) {
                        selector = TankerTextGenerator::selectEnglishNumberForm;
                    }
                    byLang.put(language, new FormSelector(new FormsData(forms,
                            numberParameterName0.get().substring(TextProcessor.ATTRIBUTES_PREFIX.length())), selector));
                }

                if (byLang.size() == Language.values().length) {
                    data.put(keysetName + "/" + key, byLang);
                }
            }
        }

        return data;
    }

    private static String selectRussianAndUkrainianNumberForm(Object value, FormsData formsData) {
        if (!(value instanceof Integer)) {
            throw new IllegalArgumentException("Wrong type of number value");
        }
        int number = (Integer) value;
        if (number == 1) {
            return formsData.getForms().get(3);
        }
        if (number % 10 == 1 && number % 100 != 11) {
            return formsData.getForms().first();
        }
        if (RUSSIAN_SECOND_FORM_MODS_VALUES.containsTs(number % 10) &&
                !RUSSIAN_NON_SECOND_FORM_MODS_VALUES.containsTs(number % 100)) {
            return formsData.getForms().get(1);
        }
        return formsData.getForms().get(2);
    }

    private static String selectEnglishNumberForm(Object value, FormsData formsData) {
        if (!(value instanceof Integer)) {
            throw new IllegalArgumentException("Wrong type of number value");
        }
        int number = (Integer) value;
        if (number == 1) {
            return formsData.getForms().get(3);
        }
        return formsData.getForms().get(1);
    }

    public TankerTextGenerator(InputStreamSource config, TextProcessor textProcessor) {
        this.tankerConfiguration = parseTankerFormat(config);
        this.textProcessor = textProcessor;
    }

    public LocalizedString processOneFromSetOrTemplatesByKeyPrefix(String templateKeyPrefix,
            TitleGenerationContext context)
    {
        Pattern pattern = Pattern.compile(templateKeyPrefix + "_v\\d+$");
        return processOneFromSetOfTemplates(tankerConfiguration.keys().filter(key -> pattern.matcher(key).matches()),
                context);
    }

    public LocalizedString processOneFromSetOfTemplates(ListF<String> templateKeys, TitleGenerationContext context) {
        return processTemplateByKey(context.getR().randomElement(templateKeys), context);
    }

    public LocalizedString processTemplateByKey(String templateKey, TitleGenerationContext context) {
        MapF<Language, FormSelector> textsConfiguration = tankerConfiguration.getOrThrow(templateKey,
                String.format("The template key `%s` is invalid", templateKey));
        MapF<Language, String> selectedForms = textsConfiguration.mapValues(
                selector -> selector.selectFormForValue(
                        context.getAttributes().getTs(selector.getFormsData().getSelectorParameterName())
                ));
        ListF<Language> languages = BlockTitlesGenerator.SUPPORTED_LANGUAGES;
        return new DynamicLocalizedString(languages.toMap(lang -> Tuple2.tuple(lang,
                textProcessor.processTemplate(selectedForms.getTs(lang), lang, context))));
    }

    public boolean containsAnyVersionOfKey(String keyPrefix) {
        Pattern pattern = Pattern.compile(keyPrefix + "_v\\d+$");
        return !tankerConfiguration.keys().filter(key -> pattern.matcher(key).matches()).isEmpty();
    }

    @Data
    private static class FormSelector {
        private final FormsData formsData;
        private final BiFunction<Object, FormsData, String> selector;

        public String selectFormForValue(Object value) {
            return selector.apply(value, formsData);
        }
    }

    @Data
    private static class FormsData {
        private final ListF<String> forms;
        private final String selectorParameterName;
    }
}
