package ru.yandex.chemodan.app.lentaloader.cool.utils;

import org.joda.time.DateTime;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.utils.Language;
import ru.yandex.inside.utils.LocalizedString;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

public class TankerTextGeneratorTest {
    private static final String YEARS_ATTRIBUTE_NAME = TextProcessor.NUMBER_ATTRIBUTES_PREFIX + "years";

    private TankerTextGenerator generator;

    @Before
    public void initGenerator() {
        generator = new TankerTextGenerator(new ClassPathResourceInputStreamSource(getClass(),
                "tanker.test.json"), new TextProcessor(GeoNamesSource.EMPTY));
    }

    @Test
    public void testOneYear() {
        LocalizedString firstResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v1"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 1)));
        Assert.assertTrue("В этот день год назад".equals(firstResult.get(Language.RUSSIAN)));
        Assert.assertTrue("У цей день рік назад".equals(firstResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("On this day last year".equals(firstResult.get(Language.ENGLISH)));
        Assert.assertTrue("1 yıl önce bugün".equals(firstResult.get(Language.TURKISH)));
        LocalizedString secondResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v2"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 1)));
        Assert.assertTrue("Ровно год назад".equals(secondResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Рівно рік назад".equals(secondResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("Exactly 1 year ago".equals(secondResult.get(Language.ENGLISH)));
        Assert.assertTrue("Tam 1 yıl önce".equals(secondResult.get(Language.TURKISH)));
    }

    @Test
    public void testModeOneYear() {
        LocalizedString firstResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v1"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 21)));
        Assert.assertTrue("В этот день 21 год назад".equals(firstResult.get(Language.RUSSIAN)));
        Assert.assertTrue("У цей день 21 рік назад".equals(firstResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("On this day 21 years ago".equals(firstResult.get(Language.ENGLISH)));
        Assert.assertTrue("21 yıl önce bugün".equals(firstResult.get(Language.TURKISH)));
        LocalizedString secondResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v2"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 21)));
        Assert.assertTrue("Ровно 21 год назад".equals(secondResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Рівно 21 рік назад".equals(secondResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("Exactly 21 years ago".equals(secondResult.get(Language.ENGLISH)));
        Assert.assertTrue("Tam 21 yıl önce".equals(secondResult.get(Language.TURKISH)));
    }

    @Test
    public void testTwoYears() {
        LocalizedString firstResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v1"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 2)));
        Assert.assertTrue("В этот день 2 года назад".equals(firstResult.get(Language.RUSSIAN)));
        Assert.assertTrue("У цей день 2 роки назад".equals(firstResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("On this day 2 years ago".equals(firstResult.get(Language.ENGLISH)));
        Assert.assertTrue("2 yıl önce bugün".equals(firstResult.get(Language.TURKISH)));
        LocalizedString secondResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v2"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 2)));
        Assert.assertTrue("Ровно 2 года назад".equals(secondResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Рівно 2 роки назад".equals(secondResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("Exactly 2 years ago".equals(secondResult.get(Language.ENGLISH)));
        Assert.assertTrue("Tam 2 yıl önce".equals(secondResult.get(Language.TURKISH)));
    }

    @Test
    public void testManyYears() {
        LocalizedString firstResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v1"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 11)));
        Assert.assertTrue("В этот день 11 лет назад".equals(firstResult.get(Language.RUSSIAN)));
        Assert.assertTrue("У цей день 11 років назад".equals(firstResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("On this day 11 years ago".equals(firstResult.get(Language.ENGLISH)));
        Assert.assertTrue("11 yıl önce bugün".equals(firstResult.get(Language.TURKISH)));
        LocalizedString secondResult = generator.processOneFromSetOfTemplates(Cf.list("notifier-notifications/cool_lenta_n_years_ago_cover_title_v2"),
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 13)));
        Assert.assertTrue("Ровно 13 лет назад".equals(secondResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Рівно 13 років назад".equals(secondResult.get(Language.UKRAINIAN)));
        Assert.assertTrue("Exactly 13 years ago".equals(secondResult.get(Language.ENGLISH)));
        Assert.assertTrue("Tam 13 yıl önce".equals(secondResult.get(Language.TURKISH)));
    }

    @Test
    public void testPrefix() {
        LocalizedString result = generator.processOneFromSetOrTemplatesByKeyPrefix("notifier-notifications/cool_lenta_n_years_ago_cover_title",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map(YEARS_ATTRIBUTE_NAME, 4)));
        System.out.println(result.get(Language.RUSSIAN));
        Assert.assertTrue(Cf.set("В этот день 4 года назад", "Ровно 4 года назад").containsTs(result.get(Language.RUSSIAN)));
        Assert.assertTrue(Cf.set("On this day 4 years ago", "Exactly 4 years ago").containsTs(result.get(Language.ENGLISH)));
        Assert.assertTrue(Cf.set("4 yıl önce bugün", "Tam 4 yıl önce").containsTs(result.get(Language.TURKISH)));
        Assert.assertTrue(Cf.set("У цей день 4 роки назад", "Рівно 4 роки назад").containsTs(result.get(Language.UKRAINIAN)));
    }

    @Test
    public void testOneFormAttribute() {
        LocalizedString result = generator.processTemplateByKey("notifier-notifications/test_one_form_value",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.empty(), Cf.map("test", 1)));
        Assert.assertTrue("Ровно 1 год назад".equals(result.get(Language.RUSSIAN)));
        Assert.assertTrue("Exactly 1 year ago".equals(result.get(Language.ENGLISH)));
        Assert.assertTrue("Tam 1 yıl önce".equals(result.get(Language.TURKISH)));
        Assert.assertTrue("Рівно 1 рік назад".equals(result.get(Language.UKRAINIAN)));
    }
    @Test

    public void testTerm() {
        LocalizedString genitiveResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v1",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Ваши фотографии природы".equals(genitiveResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Fotoğraflarınız doğa".equals(genitiveResult.get(Language.TURKISH)));

        LocalizedString altNominativeResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v3",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Природа в природе".equals(altNominativeResult.get(Language.RUSSIAN)));

        LocalizedString prepositionalResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v7",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Ваша история о природе".equals(prepositionalResult.get(Language.RUSSIAN)));

        LocalizedString ablativeResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v6",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Полюбоваться природой".equals(ablativeResult.get(Language.RUSSIAN)));

        LocalizedString accusativeResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v13",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Снова посмотрите на природу".equals(accusativeResult.get(Language.RUSSIAN)));

        LocalizedString nominativeResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v4",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Фотоподборка «Природа»".equals(nominativeResult.get(Language.RUSSIAN)));
        Assert.assertTrue("Fotoğraf serisi \"tabiat\"".equals(nominativeResult.get(Language.TURKISH)));

        LocalizedString whereAdverbResult = generator.processTemplateByKey("notifier-notifications/thematic_blocks_text_v12",
                new TitleGenerationContext(new Random2(), IntervalType.ONE_DAY, DateTime.now(), DateTime.now(),
                        Cf.list(), Option.of(createTermDefinition()), Cf.map()));
        Assert.assertTrue("Вернитесь на природу".equals(whereAdverbResult.get(Language.RUSSIAN)));
    }

    private TermDefinition createTermDefinition() {
        TermLanguageDefinition russianLanguageDefinition = new TermLanguageDefinition("природа", "природы", "о", "природе",
                "природой", "природу", "на природу", Option.of("природа в природе"));
        TermLanguageDefinition turkishLanguageDefinition = new TermLanguageDefinition("tabiat", "doğa", "doğa", "hakkında",
                "", "", "", Option.empty());
        TermLanguageDefinition englishLanguageDefinition = new TermLanguageDefinition("nature", "of nature", "of", "nature",
                "", "", "", Option.empty());
        TermLanguageDefinition ukrainianLanguageDefinition = new TermLanguageDefinition("природа", "природи", "про",
                "природу", "", "", "", Option.empty());
        return new TermDefinition(Cf.map(Language.RUSSIAN, russianLanguageDefinition,
                Language.TURKISH, turkishLanguageDefinition, Language.ENGLISH, englishLanguageDefinition,
                Language.UKRAINIAN, ukrainianLanguageDefinition));
    }

}
