package ru.yandex.chemodan.app.lentaloader.cool.utils;

import java.text.SimpleDateFormat;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.joda.time.DateTime;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.inside.utils.Language;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.io.ClassPathResourceInputStreamSource;
import ru.yandex.misc.lang.StringUtils;

public class TextProcessor {

    public static final String ATTRIBUTES_PREFIX = "attributes:";

    public static final String TERM_ATTRIBUTE_PREFIX = "term.";

    public static final String YEAR_ATTRIBUTE_PREFIX = "year.";

    public static final String NUMBER_ATTRIBUTES_PREFIX = "n.";

    private static final Pattern ATTRIBUTES_PATTERN = Pattern.compile("\\{\\{([^/][^}]*)}}");

    private static final ListF<String> CAPITALIZERS = Cf.list(".", "«");

    private static final CommonTextGenerator.CommonModel common = Bender.jsonParser(CommonTextGenerator.CommonModel.class)
            .parseJson(new ClassPathResourceInputStreamSource(CommonTextGenerator.class, "common.json"));

    private static final MapF<Language, String> concatenators = Cf.map(
            Language.RUSSIAN, " и ",
            Language.ENGLISH, " and ",
            Language.UKRAINIAN, " i ",
            Language.TURKISH, " ve "
    );

    private static final MapF<String, Function<TermLanguageDefinition, String>> TERM_FORMS_PROCESSORS =
            Cf.<String, Function<TermLanguageDefinition, String>>map(
                    "nominative", TermLanguageDefinition::getNominative,
                    "genitive", TermLanguageDefinition::getGenitive,
                    "prepositional", termLanguageDefinition ->
                            String.format("%s %s", termLanguageDefinition.getPreposition(), termLanguageDefinition.getPrepositionalCase())
                                    .trim())
                    .plus(Cf.map(
                            "ablative", TermLanguageDefinition::getAblative,
                            "accusative", TermLanguageDefinition::getAccusative,
                            "whereAdverb", TermLanguageDefinition::getWhereAdverbForm,
                            "altNominative", termLanguageDefinition ->
                                    termLanguageDefinition.getAltNominative().getOrElse(termLanguageDefinition.getNominative())));

    /**
     *
     * @param forms
     * @return name of the number parameter
     */
    public static Option<String> findNumberParameterInForms(ListF<String> forms) {
        Option<String> parameterNameO = findNumberParameterInForm(forms.first());
        if (!parameterNameO.isPresent()) {
            return Option.empty();
        }
        boolean hasFormWithoutParameter = false;
        String parameterName = parameterNameO.get();
        for (int i = 1; i < forms.size(); i++) {
            String form = forms.get(i);
            if (formContainsParameter(form, parameterName)) {
                continue;
            }
            if (hasFormWithoutParameter) {
                return Option.empty();
            }
            hasFormWithoutParameter = true;
        }
        return parameterNameO;
    }

    private static Option<String> findNumberParameterInForm(String form) {
        Matcher matcher = ATTRIBUTES_PATTERN.matcher(form);
        while (matcher.find()) {
            String parameterName = matcher.group(1);
            if (parameterName.startsWith(ATTRIBUTES_PREFIX + NUMBER_ATTRIBUTES_PREFIX)) {
                return Option.of(parameterName);
            }
        }
        return Option.empty();
    }

    private static boolean formContainsParameter(String form, String parameterName) {
        Matcher matcher = ATTRIBUTES_PATTERN.matcher(form);
        while (matcher.find()) {
            if (parameterName.equals(matcher.group(1))) {
                return true;
            }
        }
        return false;
    }

    private final GeoNamesSource geoNamesSource;

    public TextProcessor(GeoNamesSource geoNamesSource) {
        this.geoNamesSource = geoNamesSource;
    }

    public boolean isSupportedSimpleTemplate(String template) {
        return template.startsWith("date.start:")
                || template.startsWith("date.end:")
                || template.startsWith("month.start:")
                || template.startsWith("month.end:")
                || template.startsWith("season.start:")
                || template.startsWith("season.end:")
                || template.startsWith("geo:")
                || template.startsWith(ATTRIBUTES_PREFIX)
                || template.startsWith(TERM_ATTRIBUTE_PREFIX)
                || template.startsWith(YEAR_ATTRIBUTE_PREFIX)
                ;
    }

    public String processTemplate(String template, Language language, TitleGenerationContext context) {
        Matcher matcher = ATTRIBUTES_PATTERN.matcher(template);
        StringBuilder sb = new StringBuilder();
        int pos = 0;
        while (matcher.find()) {
            sb.append(template, pos, matcher.start());
            pos = matcher.end();
            String processedTemplate = processSimpleTemplate(matcher.group(1), language, context);
            String stripped = StringUtils.strip(sb.toString());
            if (matcher.start() == 0 || CAPITALIZERS.find(capitalizer -> stripped.endsWith(capitalizer)).isPresent()) {
                processedTemplate = StringUtils.capitalize(processedTemplate);
            }
            sb.append(processedTemplate);
        }
        sb.append(template.substring(pos));
        return sb.toString();
    }

    public String processSimpleTemplate(String template, Language language, TitleGenerationContext context) {
        if (!isSupportedSimpleTemplate(template)) {
            throw new IllegalStateException("Unknown template " + template);
        }

        // Хак чтобы для диапазонов дат было правильное окончание
        DateTime end = context.end.minusSeconds(1);
        DateTime start = context.start;

        try {
            if (template.startsWith("date.start:")) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(template.substring("date.start:".length()));
                simpleDateFormat.setTimeZone(start.getZone().toTimeZone());
                return simpleDateFormat.format(start.toDate());
            }
            if (template.startsWith("date.end:")) {
                SimpleDateFormat simpleDateFormat = new SimpleDateFormat(template.substring("date.end:".length()));
                simpleDateFormat.setTimeZone(end.getZone().toTimeZone());
                return simpleDateFormat.format(end.toDate());
            } else if (template.startsWith("month.start:")) {
                String caseName = template.substring("month.start:".length());
                return common.monthsByLangByCase
                        .get(start.getMonthOfYear() - 1)
                        .getTs(language.value())
                        .getOrThrow(caseName, "unknown case: " + caseName);
            } else if (template.startsWith("month.end:")) {
                String caseName = template.substring("month.end:".length());
                return common.monthsByLangByCase
                        .get(end.getMonthOfYear() - 1)
                        .getTs(language.value())
                        .getOrThrow(caseName, "unknown case: " + caseName);
            } else if (template.startsWith("season.start:")) {
                Season season = TimeIntervalUtils.getSeason(start);
                String caseName = template.substring("season.start:".length());
                return common.seasonsByLangByCase
                        .get(season.ordinal())
                        .getTs(language.value())
                        .getOrThrow(caseName, "unknown case: " + caseName);
            } else if (template.startsWith("season.end:")) {
                Season season = TimeIntervalUtils.getSeason(end);
                String caseName = template.substring("season.end:".length());
                return common.seasonsByLangByCase
                        .get(season.ordinal())
                        .getTs(language.value())
                        .getOrThrow(caseName, "unknown case: " + caseName);
            } else if (template.startsWith(YEAR_ATTRIBUTE_PREFIX)) {
                String parameter = template.substring(YEAR_ATTRIBUTE_PREFIX.length());
                if ("season".equals(parameter)) {
                    int seasonStartYear = TimeIntervalUtils.getSeasonStart(start).getYear();
                    int seasonEndYear = TimeIntervalUtils.getSeasonEnd(start).getYear();
                    if (seasonStartYear == seasonEndYear) {
                        return String.valueOf(seasonStartYear);
                    }
                    return seasonStartYear + "–" + seasonEndYear;
                } else {
                    throw new IllegalArgumentException("Incorrect parameter for year template");
                }
            } else if (template.startsWith("geo:")) {
                String tail = template.substring("geo:".length());
                String[] parts = tail.split("\\.");

                int geoIndex = 0;
                GeoNamesSource.Case geoCase;
                if (parts.length > 1) {
                    if (parts[0].equals("all")) {
                        geoIndex = -1;
                    } else {
                        geoIndex = Integer.valueOf(parts[0]);
                    }
                    geoCase = GeoNamesSource.Case.R.valueOf(parts[1]);
                } else {
                    geoCase = GeoNamesSource.Case.R.valueOf(parts[0]);
                }

                if(context.geoIds.length() <= geoIndex) {
                    throw new IllegalArgumentException("Not enough geo ids: ids=" + context.geoIds.mkString(",") + ", index=" + geoIndex);
                }

                if (geoIndex == -1) {
                    return context.geoIds
                            .map(regionId -> geoNamesSource.getGeoName(regionId, geoCase, language))
                            .mkString(concatenators.getTs(language));
                } else {
                    return geoNamesSource.getGeoName(context.geoIds.get(geoIndex), geoCase, language);
                }
            } else if (template.startsWith(TERM_ATTRIBUTE_PREFIX)) {
                String termForm = template.substring(TERM_ATTRIBUTE_PREFIX.length());
                Function<TermLanguageDefinition, String> termFormProcessor = TERM_FORMS_PROCESSORS
                        .getOrThrow(termForm,
                                String.format("Unknown form for term `%s`", termForm));
                return context.getTerm().getOrThrow("Empty value for term").getLanguageTerms()
                        .getO(language).map(termFormProcessor)
                        .getOrThrow(String.format("The term is not defined for the language `%s`", language));
            } else if (template.startsWith(ATTRIBUTES_PREFIX)) {
                String attribute = template.substring(ATTRIBUTES_PREFIX.length());
                return context.getAttributes()
                        .getOrThrow(attribute, String.format("Empty value for parameter `%s`", template)).toString();
            } else  {
                throw new IllegalStateException("Unknown template " + template);
            }
        } catch (RuntimeException e) {
            throw new IllegalStateException(
                    StringUtils.format("Failed to process template={} for lang={}, start={}, end={}", template, language, start, end), e);
        }
    }
}
