package ru.yandex.chemodan.app.lentaloader.cool.utils;

import org.joda.time.DateTime;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.lang.Check;

/**
 * @author tolmalev
 */
public class TimeIntervalUtils {

    public static DateTime getDayStart(DateTime time) {
        return time.withTimeAtStartOfDay();
    }

    public static DateTime getDayEnd(DateTime time) {
        return getDayStart(time).plusDays(1);
    }

    public static DateTime getWeekendStart(DateTime time) {
        Check.isTrue(isInWeekend(time), "Not in weekend");
        DateTime dayStart = getDayStart(time);

        if (dayStart.getDayOfWeek() == 7) {
            dayStart = dayStart.minusDays(1);
        }
        return dayStart;
    }

    public static DateTime getWeekendEnd(DateTime time) {
        return getWeekendStart(time).plusDays(2);
    }

    public static DateTime getWeekStart(DateTime time) {
        return getDayStart(time).minusDays(time.getDayOfWeek() - 1);
    }

    public static DateTime getWeekEnd(DateTime time) {
        return getWeekStart(time).plusDays(7);
    }

    public static boolean isInWeekend(DateTime time) {
        int dayOfWeek = time.getDayOfWeek();

        return dayOfWeek == 6 || dayOfWeek == 7;
    }

    public static DateTime getMonthStart(DateTime time) {
        return getDayStart(time).minusDays(time.getDayOfMonth() - 1);
    }

    public static DateTime getMonthEnd(DateTime time) {
        return getMonthStart(time).plusMonths(1);
    }

    public static DateTime getYearStart(DateTime time) {
        return getDayStart(time).minusDays(time.getDayOfYear() - 1);
    }

    public static DateTime getYearEnd(DateTime time) {
        return getYearStart(time).plusYears(1);
    }

    public static DateTime getSeasonStart(DateTime time) {
        Season season = getSeason(time);
        time = getMonthStart(time);
        int firstMonth = season.getMonths().first();
        while (time.getMonthOfYear() != firstMonth) {
            time = time.minusMonths(1);
        }
        return time;
    }

    public static DateTime getSeasonEnd(DateTime time) {
        return getSeasonStart(time).plusMonths(3);
    }

    public static Season getSeason(DateTime time) {
        int month = time.getMonthOfYear();
        return Cf.x(Season.values()).find(s -> s.getMonths().containsTs(month)).single();
    }

    public static IntervalType getMinimalInterval(DateTime start, DateTime end) {
        for (IntervalType type : IntervalType.values()) {
            if (type != IntervalType.WEEKEND ||
                    TimeIntervalUtils.isInWeekend(start) && TimeIntervalUtils.isInWeekend(end.minusSeconds(1)))
            {
                DateTime intervalStart = type.getIntervalStart(start);
                DateTime intervalEnd = type.getIntervalEnd(start);

                if ((intervalStart.isBefore(end) || intervalStart.isEqual(end)) && (intervalEnd.isAfter(end) || intervalEnd.isEqual(end))) {
                    return type;
                }
            }
        }
        return IntervalType.WHOLE_TIME;
    }

    public static boolean isIntervalFinished(IntervalType type, DateTime intervalStart) {
        return isIntervalFinished(type, intervalStart, Instant.now());
    }

    public static boolean isIntervalFinished(IntervalType type, DateTime intervalStart, Instant currentTime) {
        DateTime intervalEnd = type.getIntervalEnd(intervalStart);
        return intervalEnd.isBefore(currentTime) || intervalEnd.getMillis() == currentTime.getMillis();
    }
}
