package ru.yandex.chemodan.app.lentaloader.cool.utils;

import org.joda.time.DateTime;
import org.joda.time.DateTimeZone;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
public class TimeIntervalUtilsTest {
    @Test
    public void getDayStart() {
        checkEquals(
                DateTime.parse("2007-12-03T00:00:00.00+0300"),
                TimeIntervalUtils.getDayStart(DateTime.parse("2007-12-03T10:15:30.00+0300"))
        );

        checkEquals(
                DateTime.parse("2007-12-03T00:00:00.00+0000"),
                TimeIntervalUtils.getDayStart(DateTime.parse("2007-12-03T10:15:30.00+0000"))
        );

        checkNotEquals(
                DateTime.parse("2007-12-03T00:00:00.00+0200"),
                TimeIntervalUtils.getDayStart(DateTime.parse("2007-12-03T10:15:30.00+0000"))
        );
    }

    @Test
    public void getDayEnd() {
        checkEquals(
                DateTime.parse("2007-12-04T00:00:00.00+0300"),
                TimeIntervalUtils.getDayEnd(DateTime.parse("2007-12-03T10:15:30.00+0300"))
        );

        checkEquals(
                DateTime.parse("2007-12-04T00:00:00.00+0000"),
                TimeIntervalUtils.getDayEnd(DateTime.parse("2007-12-03T10:15:30.00+0000"))
        );

        checkNotEquals(
                DateTime.parse("2007-12-04T00:00:00.00+0200"),
                TimeIntervalUtils.getDayEnd(DateTime.parse("2007-12-03T10:15:30.00+0000"))
        );
    }

    @Test
    public void getWeekendStart() {
        checkEquals(
                DateTime.parse("2018-10-27T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendStart(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-27T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendStart(DateTime.parse("2018-10-27T00:00:05.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-27T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendStart(DateTime.parse("2018-10-27T01:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-27T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendStart(DateTime.parse("2018-10-27T23:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-27T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendStart(DateTime.parse("2018-10-28T01:00:00.00+0300"))
        );
    }

    @Test
    public void getWeekendEnd() {
        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendEnd(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendEnd(DateTime.parse("2018-10-27T00:00:05.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendEnd(DateTime.parse("2018-10-27T01:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendEnd(DateTime.parse("2018-10-27T23:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekendEnd(DateTime.parse("2018-10-28T01:00:00.00+0300"))
        );
    }

    @Test
    public void getWeekStart() {
        checkEquals(
                DateTime.parse("2018-10-22T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekStart(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-22T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekStart(DateTime.parse("2018-10-27T00:00:05.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-22T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekStart(DateTime.parse("2018-10-27T01:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-22T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekStart(DateTime.parse("2018-10-27T23:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-22T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekStart(DateTime.parse("2018-10-28T01:00:00.00+0300"))
        );
    }

    @Test
    public void getWeekEnd() {
        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekEnd(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekEnd(DateTime.parse("2018-10-27T00:00:05.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekEnd(DateTime.parse("2018-10-27T01:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekEnd(DateTime.parse("2018-10-27T23:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-29T00:00:00.00+0300"),
                TimeIntervalUtils.getWeekEnd(DateTime.parse("2018-10-28T01:00:00.00+0300"))
        );
    }

    @Test
    public void getMonthStart() {
        checkEquals(
                DateTime.parse("2018-10-01T00:00:00.00+0300"),
                TimeIntervalUtils.getMonthStart(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-10-01T00:00:00.00+0000"),
                TimeIntervalUtils.getMonthStart(DateTime.parse("2018-10-27T00:00:00.00+0000"))
        );

        checkEquals(
                DateTime.parse("2020-02-01T00:00:00.00+0300"),
                TimeIntervalUtils.getMonthStart(DateTime.parse("2020-02-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getMonthEnd() {
        checkEquals(
                DateTime.parse("2018-11-01T00:00:00.00+0300"),
                TimeIntervalUtils.getMonthEnd(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-11-01T00:00:00.00+0000"),
                TimeIntervalUtils.getMonthEnd(DateTime.parse("2018-10-27T00:00:00.00+0000"))
        );

        checkEquals(
                DateTime.parse("2020-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getMonthEnd(DateTime.parse("2020-02-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getYearStart() {
        checkEquals(
                DateTime.parse("2018-01-01T00:00:00.00+0300"),
                TimeIntervalUtils.getYearStart(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-01-01T00:00:00.00+0000"),
                TimeIntervalUtils.getYearStart(DateTime.parse("2018-10-27T00:00:00.00+0000"))
        );

        checkEquals(
                DateTime.parse("2020-01-01T00:00:00.00+0300"),
                TimeIntervalUtils.getYearStart(DateTime.parse("2020-02-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getYearEnd() {
        checkEquals(
                DateTime.parse("2019-01-01T00:00:00.00+0300"),
                TimeIntervalUtils.getYearEnd(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2019-01-01T00:00:00.00+0000"),
                TimeIntervalUtils.getYearEnd(DateTime.parse("2018-10-27T00:00:00.00+0000"))
        );

        checkEquals(
                DateTime.parse("2021-01-01T00:00:00.00+0300"),
                TimeIntervalUtils.getYearEnd(DateTime.parse("2020-02-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getSeason() {
        Assert.equals(
                Season.WINTER,
                TimeIntervalUtils.getSeason(DateTime.parse("2017-12-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.WINTER,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-01-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.WINTER,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-02-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.WINTER,
                TimeIntervalUtils.getSeason(DateTime.parse("2020-02-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.WINTER,
                TimeIntervalUtils.getSeason(DateTime.parse("2020-02-29T00:00:00.00+0300"))
        );

        Assert.equals(
                Season.SPRING,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-03-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SPRING,
                TimeIntervalUtils.getSeason(DateTime.parse("2020-03-29T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SPRING,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-04-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SPRING,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-05-27T00:00:00.00+0300"))
        );

        Assert.equals(
                Season.SUMMER,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-06-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SUMMER,
                TimeIntervalUtils.getSeason(DateTime.parse("2020-06-29T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SUMMER,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-07-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.SUMMER,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-08-27T00:00:00.00+0300"))
        );

        Assert.equals(
                Season.AUTUMN,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-09-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.AUTUMN,
                TimeIntervalUtils.getSeason(DateTime.parse("2020-09-29T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.AUTUMN,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );
        Assert.equals(
                Season.AUTUMN,
                TimeIntervalUtils.getSeason(DateTime.parse("2018-11-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getSeasonStart() {
        checkEquals(
                DateTime.parse("2017-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2017-12-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2017-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-01-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2017-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-02-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-03-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-04-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-05-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-06-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-07-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-08-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-09-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonStart(DateTime.parse("2018-11-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void getSeasonEnd() {
        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2017-12-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-01-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-03-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-02-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-03-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-04-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-06-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-05-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-06-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-07-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-09-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-08-27T00:00:00.00+0300"))
        );

        checkEquals(
                DateTime.parse("2018-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-09-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-10-27T00:00:00.00+0300"))
        );
        checkEquals(
                DateTime.parse("2018-12-01T00:00:00.00+0300"),
                TimeIntervalUtils.getSeasonEnd(DateTime.parse("2018-11-27T00:00:00.00+0300"))
        );
    }

    @Test
    public void isInWeekend() {
        Assert.isTrue(TimeIntervalUtils.isInWeekend(DateTime.parse("2018-10-27T00:00:00.00+0300")));

        Assert.isTrue(TimeIntervalUtils.isInWeekend(DateTime.parse("2018-10-27T01:00:00.00+0300")));

        Assert.isTrue(TimeIntervalUtils.isInWeekend(DateTime.parse("2018-10-28T00:00:00.00+0300")));

        Assert.isTrue(TimeIntervalUtils.isInWeekend(DateTime.parse("2018-10-28T01:00:00.00+0300")));

        Assert.isFalse(TimeIntervalUtils.isInWeekend(DateTime.parse("2018-10-29T00:00:00.00+0300")));
    }

    @Test
    public void isIntervalFinished() {
        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.ONE_DAY,
                DateTime.parse("2019-02-05T00:00:00.00+0300"),
                DateTime.parse("2019-02-04T00:00:00.00+0300").toInstant()
        ));

        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.ONE_DAY,
                DateTime.parse("2019-02-05T00:00:00.00+0300"),
                DateTime.parse("2019-02-05T00:00:00.00+0300").toInstant()
        ));

        Assert.isTrue(TimeIntervalUtils.isIntervalFinished(
                IntervalType.ONE_DAY,
                DateTime.parse("2019-02-05T00:00:00.00+0300"),
                DateTime.parse("2019-02-06T00:00:00.00+0300").toInstant()
        ));

        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.WEEK,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-03T00:00:00.00+0300").toInstant()
        ));
        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.WEEK,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-04T00:00:00.00+0300").toInstant()
        ));
        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.WEEK,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-08T00:00:00.00+0300").toInstant()
        ));
        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.WEEK,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-10T23:59:00.00+0300").toInstant()
        ));
        Assert.isTrue(TimeIntervalUtils.isIntervalFinished(
                IntervalType.WEEK,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-11T00:00:00.00+0300").toInstant()
        ));

        Assert.isFalse(TimeIntervalUtils.isIntervalFinished(
                IntervalType.MONTH,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-02-28T23:59:00.00+0300").toInstant()
        ));
        Assert.isTrue(TimeIntervalUtils.isIntervalFinished(
                IntervalType.MONTH,
                DateTime.parse("2019-02-04T00:00:00.00+0300"),
                DateTime.parse("2019-03-01T00:00:00.00+0300").toInstant()
        ));
    }

    @Test
    public void getMinimalInterval() {
        DateTime current = new DateTime(1414252800000L, DateTimeZone.forID("Asia/Krasnoyarsk"));

        Assert.equals(IntervalType.ONE_DAY, TimeIntervalUtils.getMinimalInterval(
                IntervalType.ONE_DAY.getIntervalStart(current),
                IntervalType.ONE_DAY.getIntervalEnd(current)
        ));

        Assert.equals(IntervalType.ONE_DAY, TimeIntervalUtils.getMinimalInterval(
                IntervalType.ONE_DAY.getIntervalStart(current.plusSeconds(1)),
                IntervalType.ONE_DAY.getIntervalEnd(current.plusSeconds(1))
        ));

        current = DateTime.now();
        for (Integer i : Cf.range(0, 365 * 2)) {
            DateTime time = current.plusDays(i);

            for (IntervalType type : IntervalType.getTypesForDateTime(time)) {
                DateTime start = type.getIntervalStart(time);
                DateTime end = type.getIntervalEnd(time);

                Assert.equals(type, TimeIntervalUtils.getMinimalInterval(start, end));
            }
        }

    }

    @Test
    public void getTypesForDateTime() {
        Assert.sizeIs(5, IntervalType.getTypesForDateTime(DateTime.parse("2019-02-12T00:00:00.00+0300")));

        Assert.sizeIs(6, IntervalType.getTypesForDateTime(DateTime.parse("2019-02-16T00:00:00.00+0300")));
        Assert.sizeIs(6, IntervalType.getTypesForDateTime(DateTime.parse("2019-02-17T00:00:00.00+0300")));
        Assert.sizeIs(6, IntervalType.getTypesForDateTime(DateTime.parse("2019-02-17T23:59:59.00+0300")));

        Assert.sizeIs(5, IntervalType.getTypesForDateTime(DateTime.parse("2019-02-18T00:00:00.00+0300")));
    }

    private void checkEquals(DateTime expected, DateTime actual) {
        Assert.isTrue(expected.isEqual(actual), "not equals " + expected + " != " + actual);
    }

    private void checkNotEquals(DateTime expected, DateTime actual) {
        Assert.isFalse(expected.isEqual(actual), "equals " + expected + " != " + actual);
    }
}
