/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache license, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the license for the specific language governing permissions and
 * limitations under the license.
 */
package org.apache.log4j;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamException;
import java.io.Serializable;
import java.util.Locale;

public class Level extends Priority implements Serializable {

    public static final int TRACE_INT = 5000;

    public static final Level OFF = new Level(OFF_INT, "OFF", 0);

    public static final Level FATAL = new Level(FATAL_INT, "FATAL", 0);

    public static final Level ERROR = new Level(ERROR_INT, "ERROR", 3);

    public static final Level WARN = new Level(WARN_INT, "WARN", 4);

    public static final Level INFO = new Level(INFO_INT, "INFO", 6);

    public static final Level DEBUG = new Level(DEBUG_INT, "DEBUG", 7);

    public static final Level TRACE = new Level(TRACE_INT, "TRACE", 7);

    public static final Level ALL = new Level(ALL_INT, "ALL", 7);

    private static final long serialVersionUID = 3491141966387921974L;

    protected Level(final int level, final String levelStr, final int syslogEquivalent) {
        super(level, levelStr, syslogEquivalent);
    }

    public static Level toLevel(final String sArg) {
        return toLevel(sArg, Level.DEBUG);
    }

    public static Level toLevel(final int val) {
        return toLevel(val, Level.DEBUG);
    }

    public static Level toLevel(final int val, final Level defaultLevel) {
        switch (val) {
            case ALL_INT:
                return ALL;
            case DEBUG_INT:
                return Level.DEBUG;
            case INFO_INT:
                return Level.INFO;
            case WARN_INT:
                return Level.WARN;
            case ERROR_INT:
                return Level.ERROR;
            case FATAL_INT:
                return Level.FATAL;
            case OFF_INT:
                return OFF;
            case TRACE_INT:
                return Level.TRACE;
            default:
                return defaultLevel;
        }
    }

    public static Level toLevel(final String sArg, final Level defaultLevel) {
        if (sArg == null) {
            return defaultLevel;
        }
        final String s = sArg.toUpperCase(Locale.ROOT);
        switch (s) {
            case "ALL":
                return Level.ALL;
            case "DEBUG":
                return Level.DEBUG;
            case "INFO":
                return Level.INFO;
            case "WARN":
                return Level.WARN;
            case "ERROR":
                return Level.ERROR;
            case "FATAL":
                return Level.FATAL;
            case "OFF":
                return Level.OFF;
            case "TRACE":
                return Level.TRACE;
            default:
                return defaultLevel;
        }
    }

    private void readObject(final ObjectInputStream s) throws IOException, ClassNotFoundException {
        s.defaultReadObject();
        level = s.readInt();
        syslogEquivalent = s.readInt();
        levelStr = s.readUTF();
        if (levelStr == null) {
            levelStr = "";
        }
    }

    private void writeObject(final ObjectOutputStream s) throws IOException {
        s.defaultWriteObject();
        s.writeInt(level);
        s.writeInt(syslogEquivalent);
        s.writeUTF(levelStr);
    }

    protected Object readResolve() throws ObjectStreamException {
        //
        //  if the deserialized object is exactly an instance of Level
        //
        if (getClass() == Level.class) {
            return toLevel(level);
        }
        //
        //   extension of Level can't substitute stock item
        //
        return this;
    }

}

