package ru.yandex.chemodan.app.notifier.metadata;

import lombok.EqualsAndHashCode;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.config.JsonLongFormat;
import ru.yandex.misc.bender.config.JsonSettings;
import ru.yandex.misc.bender.custom.DurationAsMillisMarshaller;
import ru.yandex.misc.bender.custom.DurationAsMillisUnmarshaller;
import ru.yandex.misc.lang.ObjectUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author buberman
 */
@BenderBindAllFields
@EqualsAndHashCode
public class MetadataWrapper {
    public static final BenderMapper mapper = new BenderMapper(new BenderConfiguration(
            new BenderSettings(
                    MembersToBind.ALL_FIELDS, new JsonSettings(false, false, false, JsonLongFormat.NUMBER)),
            CustomMarshallerUnmarshallerFactoryBuilder.cons()
                    .add(MetadataWrapper.class, new MetadataMarshaller(), new MetadataUnmarshaller())
                    .add(Duration.class, new DurationAsMillisMarshaller(), new DurationAsMillisUnmarshaller())
                    .build()
    ));

    protected static final BenderParserSerializer<MetadataWrapper> parserSerializer
            = mapper.createParserSerializer(MetadataWrapper.class);

    public final MapF<String, MetadataEntity> meta;

    public MetadataWrapper(MapF meta) {
        this.meta = meta;
    }

    public Option<MetadataEntity> getEntityFieldsO(String entityName) {
        return meta.getO(entityName);
    }

    public Option<String> getEntityField(String entityName, String field) {
        return meta.getO(entityName).flatMapO(e -> e.getO(field));
    }

    /**
     * For test use only
     */
    public Option<MetadataEntityType> getEntityType(String entityName) {
        return meta.getO(entityName).flatMapO(e -> Option.of(e.getEntityType()));
    }

    public Option<String> getLocalizedEntityField(String entityName, String field, NotifierLanguage language) {
        return meta.getO(entityName).flatMapO(
                e -> e.fields.getO(field + "-" + language.value())
                        .plus(e.fields.getO(field))
                        .findNot(StringUtils::isBlank)
        );
    }

    public void put(String entityName, String field, String value) {
        this.meta.getOrElseUpdate(entityName, () -> new MetadataEntity()).put(field, value);
    }

    public SetF<String> findTextMissingEntities(SetF<String> entities) {
        return entities.filterNot(entity -> getEntityField(entity, "text").isPresent()
                || MetadataEntity.TRANSLATED_TEXT_FIELDS.exists(field -> getEntityField(entity, field).isPresent()));
    }

    public MetadataWrapper withoutTranslatedTexts() {
        MapF<String, MetadataEntity> newMeta = Cf.hashMap();

        for (String key : meta.keySet()) {
            newMeta.put(key, meta.getTs(key).withoutFields(MetadataEntity.TRANSLATED_TEXT_FIELDS));
        }
        return new MetadataWrapper(newMeta);
    }

    public String toJsonString() {
        return new String(parserSerializer.getSerializer().serializeJson(this));
    }

    public static MetadataWrapper fromJsonString(String json) {
        // parser fails on manual apostrophe escapes (\\'), so this forces them as unicode (\\u0027)
        json = json.replaceAll("(?<!\\\\)\\\\'", "\\\\u0027");
        return parserSerializer.getParser().parseJson(json);
    }

    public static MetadataWrapper createEmpty() {
        return new MetadataWrapper(Cf.hashMap());
    }

    /**
     * Merges metadata with supplied one. Missing entities will be added and current ones will be merged, overwriting
     * current fields and adding new ones from supplied metadata entities.
     * Note: Fields from current metadata that are missing in other metadata will be preserved and not deleted.
     * @param otherMetadata metadata to merge with
     * @return merged metadata
     */
    public MetadataWrapper mergeWith(MetadataWrapper otherMetadata) {
        MetadataWrapper merged = this.clone();
        for (Tuple2<String, MetadataEntity> entityTuple : otherMetadata.meta.entries()) {
            Option<MetadataEntity> entityO = merged.meta.getO(entityTuple.get1());

            if (!entityO.isPresent()) {
                merged.meta.put(entityTuple.get1(), entityTuple.get2().clone());
                continue;
            }

            for (Tuple2<String, String> fieldTuple : entityTuple.get2().fields.entries()) {
                entityO.forEach(entity -> entity.fields.put(fieldTuple));
            }
        }
        return merged;
    }

    public MetadataWrapper clone() {
        MapF<String, MetadataEntity> newMetaPayload = Cf.hashMap();
        for (String key : meta.keySet()) {
            newMetaPayload.put(key, meta.getTs(key).clone());
        }
        return new MetadataWrapper(newMetaPayload);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        MetadataWrapper that = (MetadataWrapper) o;

        return meta != null ? meta.equals(that.meta) : that.meta == null;

    }

    @Override
    public int hashCode() {
        return meta != null ? meta.hashCode() : 0;
    }

    @Override
    public String toString() {
        return ObjectUtils.toString(meta);
    }
}
