package ru.yandex.chemodan.core.worker.python.onetime;

import java.io.IOException;
import java.lang.reflect.Type;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;
import org.joda.time.Duration;
import org.springframework.context.ConfigurableApplicationContext;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.ConfigurableTaskUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.concurrent.RpsLimiter;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.spring.ApplicationContextUtils;

/**
 * @author Maksim Ahramovich
 */
public class OnetimeTaskReader {

    private static final String BEAN_PREFIX = "onetimeTask";
    private static final String CONFIG_DIRECTORY_PATH = "/etc/mpfs/onetime-tasks-config";
    private static final Duration DEFAULT_TIMEOUT = Duration.standardMinutes(5);

    public static void registerConfigurableOnetimeTasksFromConfig(ConfigurableApplicationContext applicationContext) {
        ListF<ConfigurableOnetimeTask> onetimeTasks = readConfigurableTasks();
        for (ConfigurableOnetimeTask onetimeTask : onetimeTasks) {
            ApplicationContextUtils.registerSingleton(applicationContext, BEAN_PREFIX + onetimeTask.id(), onetimeTask);
        }
    }

    private static ListF<ConfigurableOnetimeTask> readConfigurableTasks() {
        Type type = new TypeToken<List<TaskConfig>>() {}.getType();

        ListF<TaskConfig> taskConfigs = Cf.arrayList();

        File2 configDir = new File2(CONFIG_DIRECTORY_PATH);
        for (File2 file : configDir.listRegularFiles().filter(f -> f.getExtension().equals("json"))) {
            try {
                List<TaskConfig> tc = new Gson().fromJson(file.asReaderSource().getInput(), type);
                if(tc == null || tc.contains(null)) {
                    throw new JsonSyntaxException("Wrong onetime task configuration file " + file);
                }
                taskConfigs.addAll(tc);
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }

        return taskConfigs.map(config -> new ConfigurableOnetimeTask(config.getTaskName(), config.getTimeout(),
                config.getCommand(), new RpsLimiter(config.getRpsLimit()), config.getThreadCount(),
                config.getQueueCount(), config.getPriorityO()));
    }


    private static class TaskConfig {
        private String taskName;
        private String command;
        private String timeout;
        private String threadCount;
        private String queueCount;
        private String rpsLimit;
        private String priority;

        public String getTaskName() {
            Validate.notEmpty(taskName);
            return taskName;
        }

        public String getCommand() {
            Validate.notEmpty(command);
            return command;
        }

        public Duration getTimeout() {
            Validate.notEmpty(timeout);
            return ConfigurableTaskUtils.parseTimeout(timeout).getOrElse(DEFAULT_TIMEOUT);
        }

        public int getThreadCount() {
            Validate.gt(Integer.parseInt(threadCount), 0);
            return Integer.parseInt(threadCount);
        }

        public int getQueueCount() {
            Validate.ge(Integer.parseInt(queueCount), 0);
            return Integer.parseInt(queueCount);
        }

        public int getRpsLimit() {
            Validate.gt(Integer.parseInt(rpsLimit), 0);
            return Integer.parseInt(rpsLimit);
        }

        public Option<Integer> getPriorityO() {
            Option<Integer> priorityO = Option.ofNullable(priority).map(Integer::parseInt);
            if (priorityO.isPresent()) {
                Validate.gt(priorityO.get().intValue(), 0);
            }
            return priorityO;
        }
    }
}
