package ru.yandex.chemodan.core.worker.tasks;

import org.jetbrains.annotations.Nullable;
import org.joda.time.Duration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.chemodan.MpfsShellUtils;
import ru.yandex.commune.bazinga.scheduler.ActiveUidBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDropType;
import ru.yandex.commune.bazinga.scheduler.ActiveUidDuplicateBehavior;
import ru.yandex.commune.bazinga.scheduler.ActiveUniqueIdentifierConverter;
import ru.yandex.commune.bazinga.scheduler.ExecutionContext;
import ru.yandex.commune.bazinga.scheduler.OnetimeTaskSupport;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.bender.MembersToBind;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderMembersToBind;
import ru.yandex.misc.concurrent.RpsLimiter;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author akirakozov
 * @author  Lev Tolmachev
 */
public abstract class RegeneratePreview extends OnetimeTaskSupport<RegeneratePreviewParameters> {
    private static final Logger logger = LoggerFactory.getLogger(RegeneratePreview.class);

    public final DynamicProperty<Integer> rpsLimit = new DynamicProperty<>("regenerate-preview-rps-limit", 10);
    public final DynamicProperty<Long> timeoutInSeconds =
            new DynamicProperty<>("regenerate-preview-timeout-in-seconds", 3L * 60);

    private final RpsLimiter rpsLimiter;

    public RegeneratePreview(RegeneratePreviewParameters parameters) {
        super(parameters);
        rpsLimiter = new RpsLimiter(rpsLimit.get());
    }

    public RegeneratePreview(Class<RegeneratePreviewParameters> parametersClass) {
        super(parametersClass);
        rpsLimiter = new RpsLimiter(rpsLimit.get());
    }

    @Override
    protected void execute(RegeneratePreviewParameters parameters, ExecutionContext context) throws Exception {
        rpsLimiter.execute(MpfsShellUtils.executeF(Cf.list(
                "/usr/sbin/mpfs-admin-update-previews-on-demand.py",
                "--uid", parameters.uid,
                "--path", parameters.path,
                "--timeout", "" + timeoutInSeconds.get(),
                "--verbose"
        )));
    }

    @Override
    public int priority() {
        return 200;
    }

    @Override
    public Duration timeout() {
        return Duration.standardMinutes(5);
    }

    @Nullable
    @Override
    public Class<? extends ActiveUniqueIdentifierConverter<?, ?>> getActiveUidConverter() {
        return UniqueIdentifierConverter.class;
    }

    @Override
    public ActiveUidBehavior activeUidBehavior() {
        return new ActiveUidBehavior(ActiveUidDropType.WHEN_FINISHED, ActiveUidDuplicateBehavior.MERGE);
    }

    public void setRpsLimit(int rpsLimit) {
        rpsLimiter.setRpsLimit(rpsLimit);
    }

    public static final class UniqueIdentifierConverter
            implements ActiveUniqueIdentifierConverter<RegeneratePreviewParameters, UidAndPath>
    {
        @Override
        public Class getActiveUniqueIdentifierClass() {
            return UidAndPath.class;
        }

        @Override
        public UidAndPath convert(RegeneratePreviewParameters parameters) {
            return new UidAndPath(parameters.uid, parameters.path);

        }
    }

    @Bendable
    @BenderMembersToBind(MembersToBind.ALL_FIELDS)
    private static class UidAndPath {
        public final String uid;
        public final String path;

        private UidAndPath(String uid, String path) {
            this.uid = uid;
            this.path = path;
        }
    }
}
