package ru.yandex.chemodan.core.worker.tasks.monitoring;

import javax.annotation.PostConstruct;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.commune.bazinga.impl.JobStatus;
import ru.yandex.commune.bazinga.impl.controller.BazingaTaskStatistics;
import ru.yandex.commune.bazinga.impl.controller.ControllerLockHolder;
import ru.yandex.commune.bazinga.impl.controller.ControllerMetrics;
import ru.yandex.commune.bazinga.impl.controller.ControllerStateReactorSupport;
import ru.yandex.commune.bazinga.impl.controller.TaskStatusStatistics;
import ru.yandex.commune.bazinga.scheduler.OnetimeTask;
import ru.yandex.commune.dynproperties.DynamicPropertyRegistry;
import ru.yandex.commune.monitoring.arquebus.ArquebusManager;
import ru.yandex.commune.monitoring.arquebus.TriggerRegistry;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.worker.DelayingWorkerThread;
import ru.yandex.misc.worker.Workers;

/**
 * @author Maksim Ahramovich
 */
public class TaskStatesMonitor extends ControllerStateReactorSupport {

    private static final Logger logger = LoggerFactory.getLogger(TaskStatesMonitor.class);

    private final Duration delay;
    private volatile DelayingWorkerThread workerThread;

    @Autowired
    protected ArquebusManager arquebusManager;
    @Autowired
    private TriggerRegistry triggerRegistry;
    @Autowired
    private DynamicPropertyRegistry dynamicPropertyRegistry;

    private ListF<OnetimeTask> tasks;
    private BazingaTaskStatistics bazingaTaskStatistics;

    private ListF<AbstractTaskTriggerCondition> conditions = Cf.arrayList();


    public TaskStatesMonitor(
            ControllerLockHolder controllerLockHolder,
            ControllerMetrics metrics,
            Duration delay,
            BazingaTaskStatistics bazingaTaskStatistics,
            ListF<OnetimeTask> tasks)
    {
        super(controllerLockHolder, metrics);
        this.delay = delay;
        this.bazingaTaskStatistics = bazingaTaskStatistics;
        this.tasks = tasks;
    }


    @PostConstruct
    public void init() {
        tasks.forEach(task -> {
            registerCondition(new OnetimeTaskFailedTriggerCondition(task.id()));
            registerCondition(new OnetimeTaskStoppedTriggerCondition(task.id()));
        });
    }

    private void registerCondition(AbstractTaskTriggerCondition condition) {
        conditions.add(condition);
        dynamicPropertyRegistry.addInstanceFields(condition);
        dynamicPropertyRegistry.addInstanceFields(condition.getTrigger());
        triggerRegistry.addTrigger(condition.getTrigger());
    }


    @Override
    protected void onBecomeMaster() {
        workerThread = Workers.newDelayingWorker(() -> execute(), getClass().getSimpleName(),
                delay, Workers.SleepMode.PERIOD, false);
        workerThread.startGracefully();
    }

    @Override
    protected void onDismissMaster() {
        if (workerThread != null) {
            workerThread.stopGracefully();
            workerThread = null;
        }
    }


    private void execute() {
        logger.debug("Start states monitoring");
        conditions.forEach(condition -> {
            MapF<JobStatus, TaskStatusStatistics> map =
                    bazingaTaskStatistics.getCachedStatusStatistics(condition.taskId);
            if (map.isNotEmpty() && !condition.checkCondition(map)) {
                logger.debug("Swich trigger for {} task and {} condition", condition.taskId, condition);
                arquebusManager.switchTrigger(true, condition.getTrigger());
            }
        });
        logger.debug("Stop states monitoring");
    }
}
