package ru.yandex.chemodan.ydb.dao;

import com.yandex.ydb.table.description.TableColumn;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;
import com.yandex.ydb.table.values.VoidValue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.lang.Validate;

/**
 * @author tolmalev
 */
public class OneTableYdbDao extends YdbDaoBase {
    private final String tableName;
    private final TableDescription tableDescription;
    protected final StructType structType;
    protected final ListF<String> hashedColumns;
    protected final CreateTableSettings createTableSettings;

    protected OneTableYdbDao(ThreadLocalYdbTransactionManager transactionManager, String tableName,
                             TableDescription tableDescription, ListF<String> hashedColumns,
                             CreateTableSettings createTableSettings)
    {
        super(transactionManager);
        this.tableName = tableName;
        this.tableDescription = tableDescription;
        this.structType = YdbTypeUtils.structTypeFromFields(tableDescription);
        this.hashedColumns = hashedColumns;
        this.createTableSettings = createTableSettings;
    }

    protected OneTableYdbDao(ThreadLocalYdbTransactionManager transactionManager, String tableName,
            TableDescription tableDescription, CreateTableSettings createTableSettings)
    {
        this(transactionManager, tableName, tableDescription, Cf.list(), createTableSettings);
    }

    protected OneTableYdbDao(ThreadLocalYdbTransactionManager transactionManager, String tableName,
                             TableDescription tableDescription)
    {
        this(transactionManager, tableName, tableDescription, new CreateTableSettings());
    }

    public String getTableName() {
        return tableName;
    }

    protected String getTableName(Option<String> index) {
        return index.isPresent()
                ? "[" + tableName + "]:" + index.get()
                : tableName;
    }

    public TableDescription getTableDescription() {
        TableDescription.Builder builder = new TableDescription.Builder();

        tableDescription.getColumns().forEach(c -> builder.addNullableColumn(
                c.getName(),
                c.getType().getKind() == Type.Kind.OPTIONAL ? c.getType().unwrapOptional() : c.getType())
        );

        builder.setPrimaryKeys(tableDescription.getPrimaryKeys());

        Cf.x(tableDescription.getIndexes()).forEach(
                index -> builder.addGlobalIndex(index.getName(), index.getColumns()));

        return builder.build();
    }

    public CreateTableSettings getCreateTableSettings() {
        return createTableSettings;
    }

    protected void insertBatch(ListF<MapF<String, Value>> list) {
        insertOrUpsertBatch(list, false, getDefaultQuerySettings());
    }

    protected void insertBatch(ListF<MapF<String, Value>> list, ExecuteDataQuerySettings querySettings) {
        insertOrUpsertBatch(list, false, querySettings);
    }

    protected void upsertBatch(ListF<MapF<String, Value>> list) {
        upsertBatch(list, getDefaultQuerySettings());
    }

    protected void upsertBatch(ListF<MapF<String, Value>> list, ExecuteDataQuerySettings querySettings) {
        insertOrUpsertBatch(list, true, querySettings);
    }

    private void insertOrUpsertBatch(ListF<MapF<String, Value>> list, boolean upsert,
                                     ExecuteDataQuerySettings querySettings)
    {
        ListF<String> allKeys = Cf.x(tableDescription.getColumns()).map(TableColumn::getName);

        ListF<MapF<String, Value>> completeList =
                list.map(m -> allKeys.zipWith(key -> m.getO(key).getOrElse(VoidValue.of())).toMap());

        ListType listType = ListType.of(structType);
        String operation = upsert ? "UPSERT" : "INSERT";
        String sql = "" +
                "DECLARE $items AS \"" + listType + "\";\n" +
                "\n" +
                operation + " INTO " + tableName + " SELECT * FROM AS_TABLE($items);";

        Params params = Params.create()
                .put("$items", listType.newValue(completeList.map(structType::newValue)));

        execute(sql, params, querySettings);
    }

    protected void update(YdbQueryMapper.YdbCondition condition, MapF<String, Object> values) {
        update(condition, values, getDefaultQuerySettings());
    }

    protected void update(YdbQueryMapper.YdbCondition condition, MapF<String, Object> values,
                          ExecuteDataQuerySettings querySettings)
    {
        update(condition, Option.empty(), values, querySettings);
    }

    protected void update(YdbQueryMapper.YdbCondition condition, String index, MapF<String, Object> values,
                          ExecuteDataQuerySettings querySettings)
    {
        update(condition, Option.of(index), values, querySettings);
    }

    protected void update(YdbQueryMapper.YdbCondition condition, Option<String> index, MapF<String, Object> values,
                          ExecuteDataQuerySettings querySettings)
    {
        Validate.notEmpty(values);
        Tuple2List<String, Object> entries = values.entries().plus(
                values.entries().filterBy1(hashedColumns::containsTs)
                        .map1(YdbUtils::getHashName).map2(YdbUtils::getHashValue).map2(PrimitiveValue::uint32)
        );

        YdbQueryMapper.DeclarationPojo declarationPojo = YdbQueryMapper.getDeclarationPojo(
                entries.get2(), condition.params.size() + 1);

        String declareSql = condition.declareSql + declarationPojo.declareSql;

        Tuple2List<String, String> columnsMatch = entries.get1().zip(declarationPojo.params.get1());

        String sql;

        if (index.isPresent()){
            ListF<String> primaryColumns = Cf.x(getTableDescription().getPrimaryKeys());
            ListF<String> columnsToUpdate = columnsMatch.map((column, param) -> param + " AS " + column);

            String columnsString = primaryColumns.plus(columnsToUpdate).mkString(", ");
            String selectSql = "$toUpdate = (\n" +
                    "    SELECT " + columnsString + " \n" +
                    "    FROM " + getTableName(index) + " \n" +
                    "   " + condition.whereSql + "\n" +
                    ");";

            sql = declareSql + selectSql + "\n UPSERT INTO " + getTableName() + " SELECT * FROM $toUpdate;";

        } else {
            String setSql = columnsMatch
                    .map((columnName, paramName) -> columnName + " = " + paramName).mkString("SET ", ", ", "");

            sql = declareSql + "UPDATE " + getTableName() + "\n" + setSql + "\n" + condition.whereSql;
        }
        MapF<String, Value<?>> params = condition.params.plus(declarationPojo.params.toMap());

        execute(sql, toParams(params), querySettings);
    }

    protected void unset(YdbQueryMapper.YdbCondition condition, SetF<String> fieldsToUnset) {
        Validate.notEmpty(fieldsToUnset);
        String unsetSql = fieldsToUnset
                .map(field -> field + " = null").mkString("SET ", ", ", "");
        String sql = condition.declareSql + "UPDATE " + tableName + "\n" + unsetSql + "\n" + condition.whereSql;
        execute(sql, condition.params);
    }
}
