package ru.yandex.chemodan.ydb.dao;

import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.transaction.TxControl;
import com.yandex.ydb.table.values.OptionalType;
import com.yandex.ydb.table.values.OptionalValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.jetbrains.annotations.NotNull;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;

/**
 * @author tolmalev
 */
public abstract class YdbDaoBase {
    private static final OptionalType OPTIONAL_STRING = OptionalType.of(PrimitiveType.string());
    private static final OptionalValue EMPTY_OPTIONAL_STRING = OPTIONAL_STRING.emptyValue();

    protected final ThreadLocalYdbTransactionManager transactionManager;

    protected YdbDaoBase(ThreadLocalYdbTransactionManager transactionManager) {
        this.transactionManager = transactionManager;
    }

    protected <T> ListF<T> queryForList(String selectSql, SqlCondition condition, YdbRowMapper<T> rowMapper) {
        return queryForList(selectSql, condition, SqlOrder.unordered(), SqlLimits.all(), rowMapper);
    }

    protected <T> ListF<T> queryForList(String selectSql, SqlCondition condition, YdbRowMapper<T> rowMapper,
                                        ExecuteDataQuerySettings querySettings)
    {
        return queryForList(selectSql, condition, SqlOrder.unordered(), SqlLimits.all(), rowMapper, querySettings);
    }

    protected <T> ListF<T> queryForList(String selectSql, SqlCondition condition, SqlOrder order, SqlLimits limits, YdbRowMapper<T> rowMapper) {
        return queryForList(selectSql, condition, order, limits, rowMapper, getDefaultQuerySettings());
    }

    protected <T> ListF<T> queryForList(String selectSql, SqlCondition condition, SqlOrder order, SqlLimits limits,
                                        YdbRowMapper<T> rowMapper, ExecuteDataQuerySettings querySettings)
    {
        YdbQueryMapper.YdbCondition ydbCondition = YdbQueryMapper.mapWhereSql(condition);

        String finalSql = getFinalSql(selectSql, order, limits, ydbCondition);
        return queryForList(finalSql, toParams(ydbCondition.params), rowMapper, querySettings);
    }

    @NotNull
    protected Params toParams(MapF<String, Value<?>> paramsMap) {
        return Params.copyOf(paramsMap);
    }

    @NotNull
    private String getFinalSql(String selectSql, SqlOrder order, SqlLimits limits, YdbQueryMapper.YdbCondition ydbCondition) {
        return ydbCondition.declareSql
                + "\n"
                + selectSql
                + "\n"
                + ydbCondition.whereSql
                + "\n"
                + order.toSql()
                + "\n"
                + limits.toMysqlLimits()
                + ";";
    }

    protected long queryForLong(String selectSql, Params params) {
        return queryForList(selectSql, params, new SingleUint64ColumnRowMapper()).first();
    }

    protected long queryForLong(String selectSql, SqlCondition condition) {
        return queryForList(selectSql, condition, SqlOrder.unordered(), SqlLimits.all(), new SingleUint64ColumnRowMapper()).first();
    }

    protected <T> ListF<T> queryForList(String sql, Params params, YdbRowMapper<T> rowMapper) {
        return queryForList(sql, params, rowMapper, getDefaultQuerySettings());
    }

    protected <T> ListF<T> queryForList(String sql, Params params, YdbRowMapper<T> rowMapper,
                                        ExecuteDataQuerySettings querySettings)
    {
        return mapList(transactionManager.executeInTransactionOrTmpSession((session, txControl) ->
                queryForList(session, txControl, sql, params, querySettings)), rowMapper);
    }

    private <T> ListF<T> mapList(DataQueryResult dataQueryResult, YdbRowMapper<T> rowMapper) {
        try {
            ResultSetReader rs = dataQueryResult.getResultSet(0);
            return mapRows(rs, rowMapper);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    protected CompletableFuture<Result<DataQueryResult>> queryForList(Session session, TxControl txControl,
            String sql, Params params, ExecuteDataQuerySettings querySettings)
    {
        try {
            return session.executeDataQuery(sql, txControl, params, querySettings);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    protected DataQueryResult execute(String sql, MapF<String, Value<?>> params) {
        return execute(sql, toParams(params));
    }

    protected DataQueryResult execute(String sql, MapF<String, Value<?>> params, ExecuteDataQuerySettings querySettings) {
        return execute(sql, toParams(params), querySettings);
    }

    protected DataQueryResult execute(String sql, Params params) {
        return execute(sql, params, getDefaultQuerySettings());
    }

    protected DataQueryResult execute(String sql, Params params,
                                      ExecuteDataQuerySettings querySettings)
    {
        return transactionManager.executeInTransactionOrTmpSession(
                (session, txControl) -> execute(session, txControl, sql, params, querySettings));
    }

    protected CompletableFuture<Result<DataQueryResult>> execute(
            Session session, TxControl txControl, String sql, Params params,
            ExecuteDataQuerySettings querySettings)
    {
        try {
            return session.executeDataQuery(sql, txControl, params, querySettings);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    protected ExecuteDataQuerySettings getDefaultQuerySettings() {
        return transactionManager.getTimeoutSettings()
                .getExecuteDataQuerySettingsWithTimeout()
                .keepInQueryCache();
    }

    @NotNull
    protected OptionalValue getOptionalStringValue(Option<String> stringO) {
        return stringO
                .map(d -> OPTIONAL_STRING.newValue(PrimitiveValue.string(d.getBytes())))
                .getOrElse(EMPTY_OPTIONAL_STRING);
    }

    private <T> ListF<T> mapRows(ResultSetReader rs, YdbRowMapper<T> rowMapper) {
        if (rs.getRowCount() == 0) {
            return Cf.list();
        }
        ListF<T> result = Cf.arrayList();
        for (int i = 0; rs.next(); i++) {
            result.add(rowMapper.mapRow(rs, i));
        }
        return result;
    }
}
