package ru.yandex.chemodan.ydb.dao;

import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collection;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.yandex.ydb.table.values.DictType;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.TupleType;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;
import lombok.Data;
import org.jetbrains.annotations.NotNull;
import org.joda.time.ReadableInstant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.bolts.collection.Tuple4;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.enums.IntEnum;
import ru.yandex.misc.enums.StringEnum;

/**
 * @author tolmalev
 */
public class YdbQueryMapper {

    public static YdbCondition mapWhereSql(SqlCondition sqlCondition) {
        return mapWhereSql(sqlCondition, 1);
    }

    public static YdbCondition mapWhereSql(SqlCondition sqlCondition, int firstParamIndex) {
        String sourceSql = sqlCondition.whereSql();

        DeclarationPojo declarationPojo = getDeclarationPojo(sqlCondition.args(), firstParamIndex);

        StringBuilder sb = new StringBuilder();
        Matcher matcher = Pattern.compile("\\?").matcher(sourceSql);
        int pos = 0;
        int idx = 0;
        while (matcher.find()) {
            sb.append(sourceSql, pos, matcher.start());
            pos = matcher.end();

            sb.append("$params_").append(idx++ + firstParamIndex);
        }
        sb.append(sourceSql.substring(pos));

        return new YdbCondition(declarationPojo.declareSql, sb.toString(), declarationPojo.params.toMap());
    }

    public static DeclarationPojo getDeclarationPojo(ListF<Object> params, int firstParamIndex) {
        ListF<Tuple2<Type, Value<?>>> args = params.map(YdbQueryMapper::rawArgToValue);
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < args.size(); i++) {
            sb
                    .append("DECLARE $params_").append(i + firstParamIndex)
                    .append(" as \"")
                    .append(args.get(i)._1.toString())
                    .append("\";\n");
        }
        String declareSql = sb.toString();

        Tuple2List<String, Value<?>> queryParams =
                args.<Value<?>>map(Tuple2::get2).zipWithIndex().invert().map1(i -> "$params_" + (i + firstParamIndex));

        return new DeclarationPojo(declareSql, queryParams);
    }

    public static Tuple2<Type, Value<?>> rawArgToValue(Object arg) {
        if (arg instanceof Collection<?>) {
            return valueTupleForList(Cf.x((Collection<?>) arg));
        } else if (arg instanceof Object[]) {
            return valueTupleForList(Cf.x(Cf.x((Object[]) arg)));
        } else if (arg instanceof Tuple2<?, ?>) {
            Tuple2<Type, Value<?>> t1 = rawArgToValue(((Tuple2<?, ?>) arg)._1);
            Tuple2<Type, Value<?>> t2 = rawArgToValue(((Tuple2<?, ?>) arg)._2);
            TupleType tupleType = TupleType.ofOwn(t1._1, t2._1);
            return new Tuple2<>(
                    tupleType,
                    tupleType.newValue(t1._2, t2._2)
            );
        } else if (arg instanceof Tuple3<?, ?, ?>) {
            Tuple2<Type, Value<?>> t1 = rawArgToValue(((Tuple3<?, ?, ?>) arg)._1);
            Tuple2<Type, Value<?>> t2 = rawArgToValue(((Tuple3<?, ?, ?>) arg)._2);
            Tuple2<Type, Value<?>> t3 = rawArgToValue(((Tuple3<?, ?, ?>) arg)._3);

            TupleType tupleType = TupleType.ofOwn(t1._1, t2._1, t3._1);
            return new Tuple2<>(
                    tupleType,
                    tupleType.newValue(t1._2, t2._2, t3._2)
            );
        } else if (arg instanceof Tuple4<?, ?, ?, ?>) {
            Tuple2<Type, Value<?>> t1 = rawArgToValue(((Tuple4<?, ?, ?, ?>) arg)._1);
            Tuple2<Type, Value<?>> t2 = rawArgToValue(((Tuple4<?, ?, ?, ?>) arg)._2);
            Tuple2<Type, Value<?>> t3 = rawArgToValue(((Tuple4<?, ?, ?, ?>) arg)._3);
            Tuple2<Type, Value<?>> t4 = rawArgToValue(((Tuple4<?, ?, ?, ?>) arg)._4);

            TupleType tupleType = TupleType.ofOwn(t1._1, t2._1, t3._1, t4._1);
            return new Tuple2<>(
                    tupleType,
                    tupleType.newValue(t1._2, t2._2, t3._2, t4._2)
            );
        } else if (arg instanceof Map<?, ?>) {
            Map<?, ?> map = (Map<?, ?>) arg;
            if (map.isEmpty()) {
                throw new IllegalStateException("Empty collections not supported");
            }

            Tuple2List<Tuple2<Type, Value<?>>, Tuple2<Type, Value<?>>> entries = Cf.x(map).entries().toTuple2List(t2 -> Tuple2.tuple(
                    rawArgToValue(t2._1),
                    rawArgToValue(t2._2)
            ));

            DictType dictType = DictType.of(entries.first()._1._1, entries.first()._2._1);
            return new Tuple2<>(
                    dictType,
                    dictType.newValueOwn(entries.toTuple2List(t2 -> Tuple2.tuple((Value)t2._1._2, (Value)t2._2._2)).toMap())
            );
        } else if (arg instanceof PrimitiveValue) {
            return primitiveValueTuple((PrimitiveValue) arg);
        } else if (arg instanceof String) {
            return primitiveValueTuple(PrimitiveValue.string(((String) arg).getBytes()));
        } else if (arg instanceof Integer) {
            return primitiveValueTuple(PrimitiveValue.int32((Integer) arg));
        } else if (arg instanceof Long) {
            return primitiveValueTuple(PrimitiveValue.int64((Long) arg));
        } else if (arg instanceof Float) {
            return primitiveValueTuple(PrimitiveValue.float32((Float) arg));
        } else if (arg instanceof Double) {
            return primitiveValueTuple(PrimitiveValue.float64((Double) arg));
        } else if (arg instanceof IntEnum) {
            IntEnum intEnum = (IntEnum) arg;
            return primitiveValueTuple(PrimitiveValue.int32(intEnum.value()));
        } else if (arg instanceof StringEnum) {
            StringEnum stringEnum = (StringEnum) arg;
            return primitiveValueTuple(PrimitiveValue.string(stringEnum.value().getBytes()));
        } else if (arg instanceof Enum) {
            Enum plainEnum = (Enum) arg;
            return primitiveValueTuple(PrimitiveValue.string(plainEnum.name().getBytes()));
        } else if (arg instanceof Boolean) {
            return primitiveValueTuple(PrimitiveValue.bool((Boolean) arg));
        } else if (arg instanceof ReadableInstant) {
            ReadableInstant dateTime = (ReadableInstant) arg;
            return primitiveValueTuple(PrimitiveValue.timestamp(Instant.ofEpochMilli(dateTime.getMillis())));
        } else if (arg instanceof java.time.Instant) {
            Timestamp timestamp = Timestamp.from((java.time.Instant) arg);
            return primitiveValueTuple(PrimitiveValue.timestamp(timestamp.toInstant()));
        } else if (arg instanceof byte[]) {
            byte[] bytes = (byte[]) arg;
            return primitiveValueTuple(PrimitiveValue.string(bytes));
        } else if (arg instanceof DataSize) {
            return primitiveValueTuple(PrimitiveValue.int64(((DataSize) arg).toBytes()));
        } else if (arg instanceof YdbPrimitiveStringCompatible) {
            YdbPrimitiveStringCompatible primitiveValueCompatible = (YdbPrimitiveStringCompatible) arg;
            return Tuple2.tuple(PrimitiveType.string(), primitiveValueCompatible.getValue());
        } else {
            throw new IllegalStateException("Not supported " + arg);
        }
    }

    @NotNull
    private static Tuple2<Type, Value<?>> valueTupleForList(CollectionF<?> arg) {
        ListF<Tuple2<Type, Value<?>>> items = arg.map(YdbQueryMapper::rawArgToValue);

        if (items.isEmpty()) {
            throw new IllegalStateException("Empty collections not supported");
        }

        ListType listType = ListType.of(items.first()._1);
        return new Tuple2<>(
                listType,
                listType.newValue(items.map(Tuple2::get2))
        );
    }

    private static Tuple2<Type, Value<?>> primitiveValueTuple(PrimitiveValue value) {
        return new Tuple2<>(value.getType(), value);
    }

    @Data
    public static class YdbCondition {
        public final String declareSql;
        public final String whereSql;
        public final MapF<String, Value<?>> params;
    }

    @Data
    public static class DeclarationPojo {
        public final String declareSql;
        public final Tuple2List<String, Value<?>> params;
    }
}
