package ru.yandex.chemodan.ydb.dao;

import java.time.Duration;
import java.util.function.Supplier;

import com.yandex.ydb.table.settings.BeginTxSettings;
import com.yandex.ydb.table.settings.CommitTxSettings;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.PrepareDataQuerySettings;
import com.yandex.ydb.table.settings.RequestSettings;
import com.yandex.ydb.table.settings.RollbackTxSettings;

/**
 * @author friendlyevil
 */
public class YdbTimeoutSettings {
    private final Supplier<Long> queryTimeoutMillis;
    private final Supplier<Long> operationTimeoutDeltaMillis;

    private final Supplier<Long> txTimeoutMillis;
    private final Supplier<Long> txOperationTimeoutDeltaMillis;

    public YdbTimeoutSettings(Supplier<Long> queryTimeoutMillis, Supplier<Long> operationTimeoutDeltaMillis,
                              Supplier<Long> txTimeoutMillis, Supplier<Long> txOperationTimeoutDeltaMillis) {
        this.queryTimeoutMillis = queryTimeoutMillis;
        this.operationTimeoutDeltaMillis = operationTimeoutDeltaMillis;
        this.txTimeoutMillis = txTimeoutMillis;
        this.txOperationTimeoutDeltaMillis = txOperationTimeoutDeltaMillis;
    }

    public YdbTimeoutSettings(Supplier<Long> queryTimeoutMillis, Supplier<Long> operationTimeoutDeltaMillis) {
        this(queryTimeoutMillis, operationTimeoutDeltaMillis, queryTimeoutMillis, operationTimeoutDeltaMillis);
    }

    public Duration getQueryTimeout() {
        return Duration.ofMillis(queryTimeoutMillis.get());
    }

    public Duration getOperationTimeoutDelta() {
        return Duration.ofMillis(operationTimeoutDeltaMillis.get());
    }

    public Duration getOperationTimeout() {
        return getQueryTimeout().minus(getOperationTimeoutDelta());
    }

    public Duration getTxTimeout() {
        return Duration.ofMillis(txTimeoutMillis.get());
    }

    public Duration getTxOperationTimeoutDelta() {
        return Duration.ofMillis(txOperationTimeoutDeltaMillis.get());
    }

    public Duration getTxOperationTimeout() {
        return getTxTimeout().minus(getTxOperationTimeoutDelta());
    }

    public <T extends RequestSettings<T>> T withQueryTimeout(T settings) {
        return settings
                .setTimeout(getQueryTimeout())
                .setOperationTimeout(getOperationTimeout());
    }

    public <T extends RequestSettings<T>> T withTxTimeout(T settings) {
        return settings
                .setTimeout(getTxTimeout())
                .setOperationTimeout(getTxOperationTimeout());
    }

    public ExecuteDataQuerySettings getExecuteDataQuerySettingsWithTimeout() {
        return withQueryTimeout(new ExecuteDataQuerySettings());
    }

    public BeginTxSettings getBeginTxSettingsWithTimeout() {
        return withTxTimeout(new BeginTxSettings());
    }

    public CommitTxSettings getCommitTxSettingsWithTimeout() {
        return withTxTimeout(new CommitTxSettings());
    }

    public RollbackTxSettings getRollbackTxSettingsWithTimeout() {
        return withTxTimeout(new RollbackTxSettings());
    }

    public PrepareDataQuerySettings getPrepareDataQuerySettingsWithTimeout() {
        return withQueryTimeout(new PrepareDataQuerySettings());
    }
}
