package ru.yandex.chemodan.ydb.dao;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import com.yandex.ydb.core.Status;
import com.yandex.ydb.table.Session;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.values.PrimitiveValue;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.digest.Md5;

/**
 * @author yashunsky
 */
public class YdbUtils {

    public static void createTablesIfMissing(String database, Session session, ListF<OneTableYdbDao> daos) {
        try {
            for (OneTableYdbDao dao : daos) {
                String path = database + "/" + dao.getTableName();

                if (!session.describeTable(path).get().isSuccess()) {
                    createTable(session, database, dao.getTableName(), dao.getTableDescription(), dao.getCreateTableSettings())
                            .get().expect("Failed to create table");
                }
            }
        } catch (InterruptedException|ExecutionException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public static void dropTableIfExists(String path, Session session) {
        try {
            if (session.describeTable(path).get().isSuccess()) {
                session.dropTable(path).get().expect("failed to drop table");
            }
        } catch (InterruptedException|ExecutionException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public static Status createTable(ThreadLocalYdbTransactionManager manager,
            String database, String tableName, TableDescription tableDescription, CreateTableSettings settings)
    {
        return manager.executeInTmpSession(
                (session) -> createTable(session, database, tableName, tableDescription, settings));
    }

    public static CompletableFuture<Status> createTable(Session session,
            String database, String tableName, TableDescription tableDescription)
    {
        return createTable(session, database, tableName, tableDescription, new CreateTableSettings());
    }

    public static CompletableFuture<Status> createTable(Session session,
            String database, String tableName, TableDescription tableDescription, CreateTableSettings settings)
    {
        String path = database + "/" + tableName;
        return session.createTable(path, tableDescription, settings);
    }

    public static String getHashName(String columnName) {
        return columnName + "_hash";
    }

    public static int getHashValue(Object value) {
        return Md5.A.digest(String.valueOf(value)).hex().hashCode();
    }

    public static ListF<String> addHashColumns(ListF<String> columns, ListF<String> hashedColumns) {
        ListF<String> columnsWithHashed = Cf.arrayList();

        for (String column : columns) {
            if (hashedColumns.containsTs(column)) {
                columnsWithHashed.add(YdbUtils.getHashName(column));
            }
            columnsWithHashed.add(column);
        }
        return columnsWithHashed;
    }

    public static ListF<PrimitiveValue> getTimestampPartitions(int partitionsCount) {
        int partitionRange = Integer.MAX_VALUE / partitionsCount - Integer.MIN_VALUE / partitionsCount + 1;
        int offset = partitionRange / 2;
        int[] partitions = new int[partitionsCount];
        partitions[0] = offset;
        for (int i=1; i < partitionsCount; i ++) {
            partitions[i] = partitions[i - 1] + partitionRange;
        }
        return Cf.x(Arrays.stream(partitions).iterator()).map(PrimitiveValue::uint32).toList();
    }
}
