package ru.yandex.chemodan.ydb.dao.pojo;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.lang.reflect.ParameterizedType;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.settings.CreateTableSettings;
import com.yandex.ydb.table.settings.TtlSettings;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Type;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Duration;
import org.joda.time.Instant;
import org.springframework.util.ClassUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.util.bender.ISOInstantUnmarshaller;
import ru.yandex.chemodan.ydb.dao.YdbPrimitiveStringCompatible;
import ru.yandex.chemodan.ydb.dao.YdbPrimitiveStringCompatibleMarshaller;
import ru.yandex.chemodan.ydb.dao.YdbUtils;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.annotation.BenderIgnore;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.custom.IntEnumByValueUnmarshaller;
import ru.yandex.misc.bender.parse.Unmarshaller;
import ru.yandex.misc.bender.serialize.Marshaller;
import ru.yandex.misc.bender.serialize.simpleType.StringMarshaller;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.enums.IntEnum;
import ru.yandex.misc.enums.IntEnumResolver;
import ru.yandex.misc.lang.CamelWords;

/**
 * @author yashunsky
 */
public class YdbClassAnalyzer {
    private static final MapF<Class, Type> PRIMITIVE_TYPES_BY_CLASS = Tuple2List.fromPairs(
            Integer.class, PrimitiveType.int32(),
            Long.class, PrimitiveType.int64(),
            Float.class, PrimitiveType.float32(),
            Double.class, PrimitiveType.float64(),
            Boolean.class, PrimitiveType.bool(),
            String.class, PrimitiveType.string(),
            Instant.class, PrimitiveType.timestamp(),
            IntEnum.class, PrimitiveType.int32(),
            Enum.class, PrimitiveType.string(),
            DataSize.class, PrimitiveType.int64()
    ).map1(c -> (Class) c).map2(t -> (Type) t).toMap();

    public static <T> Description<T> getDescription(
            Class<T> pojoClass, ListF<String> primaryFields,
            MapF<String, ListF<String>> indexes, String ttlFiled, Duration ttl)
    {
        return getDescription(pojoClass, primaryFields, indexes, Option.of(ttlFiled), Option.of(ttl),
                Tuple2List.tuple2List(), Tuple2List.tuple2List(), true);
    }

    public static <T> Description<T> getDescription(
            Class<T> pojoClass, ListF<String> primaryFields,
            MapF<String, ListF<String>> indexes)
    {
        return getDescription(pojoClass, primaryFields, indexes, Option.empty(), Option.empty(),
                Tuple2List.tuple2List(), Tuple2List.tuple2List(), true);
    }

    public static <T> Description<T> getDescription(Class<T> pojoClass, ListF<String> primaryFields,
            MapF<String, ListF<String>> indexes,
            Option<String> ttlFiled,
            Option<Duration> ttl,
            Tuple2List<Class<?>, Marshaller> customMarshallers,
            Tuple2List<Class<?>, Unmarshaller> customUnmarshallers,
            boolean hashFirstColumns)
    {
        ListF<String> hashedColumns = hashFirstColumns
                ? primaryFields.firstO().plus(indexes.values().filterMap(ListF::firstO))
                : Cf.list();

        boolean allFieldsBended = pojoClass.isAnnotationPresent(BenderBindAllFields.class);
        TableDescription.Builder tableDescriptionBuilder = TableDescription.newBuilder();
        CustomMarshallerUnmarshallerFactoryBuilder marshallerUnmarshallerFactoryBuilder =
                CustomMarshallerUnmarshallerFactoryBuilder.cons();

        marshallerUnmarshallerFactoryBuilder
                .add(Instant.class, new InstantYdbMarshaller(), new ISOInstantUnmarshaller());

        for (Field field : pojoClass.getDeclaredFields()) {
            String fieldName = resolveFieldName(field);
            if (shouldBeInDescription(allFieldsBended, field)) {
                Class clazz = ClassUtils.resolvePrimitiveIfNecessary(field.getType());

                if (Option.class.isAssignableFrom(clazz)) {
                    ParameterizedType pType = (ParameterizedType) field.getGenericType();
                    clazz = (Class)(pType.getActualTypeArguments()[0]);
                }

                if (IntEnum.class.isAssignableFrom(clazz)) {
                    marshallerUnmarshallerFactoryBuilder.add((Class<?>) clazz,
                            new IntEnumYdbMarshaller(), new IntEnumByValueUnmarshaller<>(IntEnumResolver.r(clazz)));
                    clazz = IntEnum.class;
                } else if (Enum.class.isAssignableFrom(clazz)) {
                    marshallerUnmarshallerFactoryBuilder.add((Class<?>) clazz, new StringMarshaller());
                    clazz = Enum.class;
                } else if (YdbPrimitiveStringCompatible.class.isAssignableFrom(clazz)) {
                    marshallerUnmarshallerFactoryBuilder.add((Class<?>) clazz, new YdbPrimitiveStringCompatibleMarshaller());
                    clazz = String.class;
                }

                Type type = PRIMITIVE_TYPES_BY_CLASS.getO(clazz).getOrElse(PrimitiveType.json());

                if (hashedColumns.containsTs(fieldName)) {
                    tableDescriptionBuilder.addNullableColumn(YdbUtils.getHashName(fieldName), PrimitiveType.uint32());
                }

                if (primaryFields.containsTs(fieldName)) {
                    tableDescriptionBuilder.addNonnullColumn(fieldName, type);
                } else {
                    tableDescriptionBuilder.addNullableColumn(fieldName, type);
                }
            }
        }

        customMarshallers.forEach(marshallerUnmarshallerFactoryBuilder::add);
        customUnmarshallers.forEach(marshallerUnmarshallerFactoryBuilder::add);

        indexes.forEach((name, columns) -> tableDescriptionBuilder.addGlobalIndex(name, hashFirstColumns
                ? YdbUtils.addHashColumns(columns, hashedColumns)
                : columns));

        CreateTableSettings createTableSettings = new CreateTableSettings();

        if (ttlFiled.isPresent()) {
            int ttlSecond = ttl.orElse(Duration.ZERO).toStandardSeconds().getSeconds();
            createTableSettings.setTtlSettings(new TtlSettings(ttlFiled.get(), ttlSecond));
        }

        TableDescription tableDescription = tableDescriptionBuilder
                .setPrimaryKeys(YdbUtils.addHashColumns(primaryFields, hashedColumns))
                .build();

        BenderConfiguration benderConfiguration = new BenderConfiguration(
                BenderConfiguration.defaultSettings(),
                marshallerUnmarshallerFactoryBuilder.build());

        return new Description<>(pojoClass, tableDescription, benderConfiguration, hashedColumns, createTableSettings);
    }

    private static boolean shouldBeInDescription(boolean allFieldsBended, Field field) {
        if (Modifier.isStatic(field.getModifiers())) {
            return false;
        } else if (field.isAnnotationPresent(BenderIgnore.class)) {
            return false;
        } else if (allFieldsBended) {
            return true;
        }
        return field.isAnnotationPresent(BenderPart.class);
    }

    private static String resolveFieldName(Field field) {
        if (field.isAnnotationPresent(BenderPart.class)) {
            BenderPart annotation = field.getAnnotation(BenderPart.class);
            if (annotation.strictName()) {
                return annotation.name();
            }
        }
        return CamelWords.parse(field.getName()).toDbName();
    }

    @AllArgsConstructor
    @Data
    public static class Description<T> {
        private final Class<T> pojoClass;
        private final TableDescription tableDescription;
        private final BenderConfiguration benderConfiguration;
        private final ListF<String> hashedColumns;
        private final CreateTableSettings createTableSettings;
    }

}
