package ru.yandex.chemodan.ydb.dao.pojo;

import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.Type;
import com.yandex.ydb.table.values.Value;
import lombok.AllArgsConstructor;
import lombok.Data;
import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.serialize.BenderJsonSerializer;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.enums.Color;
import ru.yandex.misc.test.Assert;

/**
 * @author yashunsky
 */
public class YdbObjectParseSerializeTest {
    private YdbClassAnalyzer.Description<SampleObject> description;
    private BenderJsonSerializer<SampleObject> serializer;
    private SampleObject sampleObject;

    @Before
    public void setup() {
        description = YdbClassAnalyzer.getDescription(SampleObject.class, Cf.list("string_value"), Cf.map(),
                Option.empty(), Option.empty(), Tuple2List.tuple2List(), Tuple2List.tuple2List(), false);
        serializer = Bender.jsonSerializer(SampleObject.class, description.getBenderConfiguration());
        sampleObject = SampleObject.some(Instant.parse("2019-07-12T11:14:00.000Z"));
    }

    @Test
    public void checkTopLevelTypes() {
        YdbObjectWriter ydbObjectWriter = new YdbObjectWriter(Cf.list());
        serializer.serializeJson(sampleObject, ydbObjectWriter);

        MapF<String, Value> result = ydbObjectWriter.getFinalResult();

        assertType(result, "string_value", PrimitiveType.Id.String);
        assertType(result, "top_level_instant", PrimitiveType.Id.Timestamp);

        assertType(result, "int_value", PrimitiveType.Id.Int32);
        assertType(result, "inner_instant", PrimitiveType.Id.Json);
        assertType(result, "array_value", PrimitiveType.Id.Json);
        assertType(result, "inner_dict", PrimitiveType.Id.Json);
        assertType(result, "simple_enum", PrimitiveType.Id.String);
        assertType(result, "color", PrimitiveType.Id.Int32);
    }

    @Test
    public void writeReadTest() {

        YdbTestUtils.doWithTable(transactionManager -> new OneTablePojoYdbDao<>(transactionManager,
                        "serializeTestTable", SampleObject.class, description),
                dao -> {
                    dao.upsert(sampleObject);
                    ListF<SampleObject> found = dao.find(SqlCondition.trueCondition());
                    Assert.equals(found.first(), sampleObject);
                });
    }

    private void assertType(MapF<String, Value> result, String field, PrimitiveType.Id id) {
        Type type = result.getTs(field).getType();
        Assert.isInstance(type, PrimitiveType.class);
        Assert.equals(id, ((PrimitiveType) type).getId());
    }

    @AllArgsConstructor
    @BenderBindAllFields
    @Data
    private static class SampleObject {
        private final String stringValue;
        private final int intValue;
        private final MapF<String, InnerObject> innerDict;
        private final ListF<Boolean> arrayValue;
        private final Instant topLevelInstant;

        private final MapF<String, Instant> innerInstant;

        private final SimpleEnum simpleEnum;

        private final Color color;

        private final Option<Integer> emptyValue;

        public static SampleObject some(Instant instant) {
            return new SampleObject(
                    "any string", 42,
                    Cf.map("f1", new InnerObject(2L, 12L), "f2", new InnerObject(8L, 50L)),
                    Cf.list(true, false), instant, Cf.map("value", instant), SimpleEnum.VALUE_A, Color.RED,
                    Option.empty());
        }

        public SampleObject withOtherIndexValue(String newStringValue) {
            return new SampleObject(newStringValue,
                    intValue, innerDict, arrayValue, topLevelInstant, innerInstant, simpleEnum, color, emptyValue);
        }
    }

    @AllArgsConstructor
    @BenderBindAllFields
    @Data
    private static class InnerObject {
        private final long v1;
        private final long v2;
    }

    private enum SimpleEnum {
        VALUE_A,
        VALUE_B
    }
}
