package ru.yandex.chemodan.ydb.dao.pojo;

import java.io.StringWriter;
import java.io.Writer;
import java.math.BigDecimal;

import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.chemodan.ydb.dao.YdbUtils;
import ru.yandex.commune.json.write.BenderJsonWriterWrapper;
import ru.yandex.commune.json.write.JsonWriter;
import ru.yandex.commune.json.write.JsonWriterImpl;
import ru.yandex.misc.bender.serialize.BenderJsonWriter;
import ru.yandex.misc.lang.CamelWords;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.Check;

/**
 * @author yashunsky
 */
public class YdbObjectWriter implements BenderJsonWriter {
    private final ListF<String> hashedColumns;
    private MapF<String, Value> finalResult;
    private Option<InnerJsonWriter> currentJsonWriter = Option.empty();
    private int level = -1;
    private ListF<String> fields = Cf.arrayList();
    private ListF<Value> values = Cf.arrayList();
    private final Function<Instant, String> instantSerializer;

    public YdbObjectWriter(ListF<String> hashedColumns, Function<Instant, String> instantSerializer) {
        this.hashedColumns = hashedColumns;
        this.instantSerializer = instantSerializer;
    }

    public YdbObjectWriter(ListF<String> hashedColumns) {
        this(hashedColumns, Instant::toString);
    }

    @Override
    public void writeObjectStart() {
        level++;
        if (level == 1) {
            currentJsonWriter = Option.of(new InnerJsonWriter());
        }
        currentJsonWriter.ifPresent(w -> w.benderJsonWriter.writeObjectStart());
    }

    @Override
    public void writeObjectEnd() {
        level--;
        currentJsonWriter.ifPresent(w -> w.benderJsonWriter.writeObjectEnd());
        if (level == 0) {
            writeJson(currentJsonWriter.get().getString());
            currentJsonWriter = Option.empty();
        } else if (level == -1) {
            Check.equals(fields.size(), values.size());
            finalResult = fields.zip(values).toMap();
        }
    }

    @Override
    public void writeFieldName(String name) {
        if (isTopLevel()) {
            fields.add(CamelWords.parse(name).toDbName());
        } else {
            currentJsonWriter.get().benderJsonWriter.writeFieldName(name);
        }
    }

    @Override
    public void writeArrayStart() {
        level++;
        if (level == 1) {
            currentJsonWriter = Option.of(new InnerJsonWriter());
        }
        currentJsonWriter.ifPresent(w -> w.benderJsonWriter.writeArrayStart());
    }

    @Override
    public void writeArrayEnd() {
        level--;
        currentJsonWriter.ifPresent(w -> w.benderJsonWriter.writeArrayEnd());
        if (level == 0) {
            writeJson(currentJsonWriter.get().getString());
            currentJsonWriter = Option.empty();
        }
    }

    @Override
    public void writeString(String value) {
        addValue(value, v -> PrimitiveValue.string(value.getBytes(CharsetUtils.UTF8_CHARSET)), BenderJsonWriter::writeString);
    }

    @Override
    public void writeNumber(int value) {
        addValue(value, PrimitiveValue::int32, BenderJsonWriter::writeNumber);
    }

    @Override
    public void writeNumber(long value) {
        addValue(value, PrimitiveValue::int64, BenderJsonWriter::writeNumber);
    }

    @Override
    public void writeNumber(double value) {
        addValue(value, PrimitiveValue::float64, BenderJsonWriter::writeNumber);
    }

    @Override
    public void writeNumber(float value) {
        addValue(value, PrimitiveValue::float32, BenderJsonWriter::writeNumber);
    }

    @Override
    public void writeNumber(BigDecimal value) {
        addValue(value, v -> PrimitiveValue.int64(v.longValue()), BenderJsonWriter::writeNumber);
    }

    @Override
    public void writeBoolean(boolean value) {
        addValue(value, PrimitiveValue::bool, BenderJsonWriter::writeBoolean);
    }

    @Override
    public void writeNull() {
        addValue(null, v -> null, (w, v) -> w.writeNull());
    }

    public void writeInstant(Instant value) {
        addValue(value,
                v -> PrimitiveValue.timestamp(java.time.Instant.ofEpochMilli(v.getMillis())), (w, v) ->
                w.writeString(instantSerializer.apply(v)));
    }

    public void writeJson(String value) {
        values.add(PrimitiveValue.json(value));
    }

    @Override
    public void flush() {
    }

    @Override
    public void close() {
    }

    public MapF<String, Value> getFinalResult() {
        return finalResult;
    }

    private boolean isTopLevel() {
        return level == 0;
    }

    private <T> void addValue(T value, Function<T, Value> converter, Function2V<BenderJsonWriter, T> innerAdd) {
        if (isTopLevel()) {
            values.add(converter.apply(value));
            String fieldName = fields.last();
            if (hashedColumns.containsTs(fieldName)) {
                fields.add(YdbUtils.getHashName(fieldName));
                values.add(PrimitiveValue.uint32(YdbUtils.getHashValue(String.valueOf(value))));
            }
        } else {
            innerAdd.apply(currentJsonWriter.get().benderJsonWriter, value);
        }
    }

    private static class InnerJsonWriter {
        private final Writer writer;
        private final JsonWriter jsonWriter;
        private final BenderJsonWriter benderJsonWriter;

        public InnerJsonWriter() {
            this.writer = new StringWriter();
            this.jsonWriter = new JsonWriterImpl(writer);
            this.benderJsonWriter = new BenderJsonWriterWrapper(jsonWriter);
        }

        public String getString() {
            return writer.toString();
        }
    }
}
