package ru.yandex.chemodan.ydb.dao.pojo;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.yandex.ydb.table.result.ValueReader;
import com.yandex.ydb.table.values.PrimitiveType;
import lombok.AllArgsConstructor;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.lang.CharsetUtils;

/**
 * @author yashunsky
 */
@AllArgsConstructor
public class YdbValueNode implements BenderJsonNode {
    private final static SetF<PrimitiveType.Id> NUMBER_IDS = Cf.set(
            PrimitiveType.Id.Int32, PrimitiveType.Id.Int64,
            PrimitiveType.Id.Float32, PrimitiveType.Id.Float64);
    private final PrimitiveType.Id typeId;
    private final ValueReader valueReader;

    @Override
    public Option<BenderJsonNode> getField(String fieldName) {
        return Option.empty();
    }

    @Override
    public boolean isNull() {
        return false;
    }

    @Override
    public boolean isObject() {
        return false;
    }

    @Override
    public boolean isNumber() {
        return NUMBER_IDS.containsTs(typeId);
    }

    @Override
    public boolean isBoolean() {
        return typeId == PrimitiveType.Id.Bool;
    }

    @Override
    public boolean isString() {
        return typeId == PrimitiveType.Id.String;
    }

    @Override
    public boolean isArray() {
        return false;
    }

    @Nonnull
    @Override
    public String getValueAsString() {
        switch (typeId) {
            case String:
                return valueReader.getString(CharsetUtils.UTF8_CHARSET);
            case Timestamp:
                return new Instant(valueReader.getTimestamp().toEpochMilli()).toString();
            case Int32:
                return String.valueOf(valueReader.getInt32());
            case Int64:
                return String.valueOf(valueReader.getInt64());
            case Float32:
                return String.valueOf(valueReader.getFloat32());
            case Float64:
                return String.valueOf(valueReader.getFloat64());
            case Bool:
                return String.valueOf(valueReader.getBool());
        }
        throw new IllegalStateException("Not supported type " + typeId);
    }

    @Nonnull
    @Override
    public String getString() {
        return valueReader.getString(CharsetUtils.UTF8_CHARSET);
    }

    @Nullable
    @Override
    public Number getNumberValueOrNull() {
        switch (typeId) {
            case Int32:
                return Integer.valueOf(String.valueOf(valueReader.getInt32()));
            case Int64:
                return Long.valueOf(String.valueOf(valueReader.getInt64()));
            case Float32:
                return Float.valueOf(String.valueOf(valueReader.getFloat32()));
            case Float64:
                return Double.valueOf(String.valueOf(valueReader.getFloat64()));
        }
        return null;
    }

    @Override
    public boolean getBooleanValueOrFalse() {
        return typeId == PrimitiveType.Id.Bool && valueReader.getBool();
    }

    @Override
    public ListF<BenderJsonNode> getArrayElements() {
        return Cf.list();
    }

    @Override
    public ListF<String> getFieldNames() {
        return Cf.list();
    }
}
