import logging
import datetime

from ci.tasklet.common.proto import service_pb2 as ci
from tasklet.services.yav.proto import yav_pb2

from disk.tasklets.conductor.proto import create_tasklet, create_pb2

import metrika.pylib.conductor.conductor as conductor
import disk.tasklets.utils.dist as dist
logger = logging.getLogger(__name__)

PACKAGES = {
    'disk': 'python-mpfs-disk',
    'platform': 'python-mpfs-api'
}


class CreateConductorTicketImpl(create_tasklet.CreateConductorTicketBase):
    def run(self):
        logger.info("Input:\n" + str(self.input))

        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.module = "CONDUCTOR"
        self.ctx.ci.UpdateProgress(progress_msg)

        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key="conductor_oauth_token")
        api = conductor.ConductorAPI(oauth_token=self.ctx.yav.get_secret(spec).secret)

        package = self.input.ticket.package or PACKAGES.get(self.input.config.package)
        if not package:
            raise Exception('Package is not set')

        if self.input.config.need_last_version:
            release_version = self.input.config.release_version
            version = dist.get_last_disk_release_on_dist(release_version)
        else:
            version = self.input.ticket.version

        if not version:
            raise Exception('Version is not set')
        logger.info("Package version: %s" % version)

        ticket = api.ticket_add(
            packages={package: version},
            branch=create_pb2.Branch.Name(self.input.ticket.branch).lower(),
            remove=self.input.ticket.packages_to_remove or None,
            mailcc=self.input.ticket.mailcc or None,
            skip_restart=self.input.ticket.skip_restart or None,
            no_autoinstall=self.input.ticket.no_autoinstall or None,
            projects=self.input.ticket.projects or None,
            comment=self.input.ticket.comment
        )

        self.output.result.name = ticket.name
        self.output.result.package_version = version

        start_ts = datetime.datetime.now()
        while datetime.datetime.now() - start_ts < datetime.timedelta(seconds=60):
            try:
                progress_msg.url = "https://c.yandex-team.ru/tickets/{}".format(self.output.result.name)
                progress_msg.text = ticket.status["status"]
                progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
                self.ctx.ci.UpdateProgress(progress_msg)
                return
            except Exception as e:
                progress_msg.url = "https://c.yandex-team.ru/tickets/{}".format(self.output.result.name)
                progress_msg.text = str(e)
                self.ctx.ci.UpdateProgress(progress_msg)
                logger.warning("Ticket not yet created", exc_info=True)

        progress_msg.url = "https://c.yandex-team.ru/tickets/{}".format(self.output.result.name)
        progress_msg.text = "Conductor ticket not created"
        progress_msg.status = ci.TaskletProgress.Status.FAILED
        self.ctx.ci.UpdateProgress(progress_msg)
        raise Exception(progress_msg.text)
