import os
import re
import logging
from configparser import ConfigParser
import subprocess
import datetime
from dateutil.tz import tzlocal

from disk.tasklets.mpfs_release.proto import create_release_branch_tasklet
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2 import ssh
from tasklet.services.yav.proto import yav_pb2

CONFIG_FILE = 'tools/builder.conf'
MPFS_DIR = 'arcadia'
RELEASE_DIR = 'release'
RELEASE_BRANCH_REGEX = re.compile(r'(?:^|/)release-(\d+)\.(\d+)$')


class CreateReleaseBranchImpl(create_release_branch_tasklet.CreateReleaseBranchBase):

    def run(self):
        arcadia_mpfs_path = 'disk/mpfs'
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, arcadia_mpfs_path)
        arcadia_dir = Arcadia.checkout(arcadia_url, MPFS_DIR)
        logging.info("Arcadia is checkouted to %s", arcadia_dir)

        login = self.input.config.login
        ssh_key = self.get_private_ssh_key()

        config = BuildConfig()
        packages = dict(config.get_packages_list()).keys()
        version = self.input.config.version

        if version == 'auto':
            major_ver, minor_ver = SVNInfo.get_last_release()
            version = '%s.%s' % (major_ver, minor_ver + 1)

        if not re.search(r'^\d\.\d+$', version):
            raise ValueError('Bad version format.')

        branch_name = "release-%s" % version
        branch_full_name = f"branches/disk/mpfs/releases/{branch_name}"
        release_url = os.path.join(Arcadia.ARCADIA_BASE_URL, branch_full_name)

        with ssh.Key(private_part=ssh_key):
            Arcadia.copy(arcadia_url, release_url, f'mpfs: new release {version}', user=login)

        release_dir = Arcadia.checkout(release_url, RELEASE_DIR)
        logging.info("Release is checkouted to %s", release_dir)

        changed_files = []

        for package in packages:
            changed_file = self.write_changelog(login, package, version)
            changed_files.append(changed_file)

        config.update(version=version, release=0)
        changed_files.append(CONFIG_FILE)

        changed_path = self.write_mpfs_version(version)
        changed_files.append(changed_path)

        logging.info(f'Gonna commit into release branch: {branch_name}')

        message = f'mpfs: Update changelog/setup/mpfs version {version}'
        with_revprop = ['arcanum:review-skip=yes', 'arcanum:check-skip=yes']

        with ssh.Key(private_part=ssh_key):
            Arcadia.commit(release_dir, message, user=login, with_revprop=with_revprop)

        self.output.release.version = version

    def get_private_ssh_key(self):
        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key='arcadia_ssh_key')
        return self.ctx.yav.get_secret(spec).secret

    def write_changelog(self, login, package, version, date={}):
        email = self.get_user_email(login)
        date_deb = date.get('deb')
        if date_deb:
            full_date = date_deb
        else:
            full_date = date['deb'] = datetime.datetime.now(tzlocal()).strftime("%a, %d %b %Y %H:%M:%S %z")
        record = 'python-mpfs-%s (%s-1) unstable; urgency=low\n\n  * Common Release\n\n -- %s <%s>  %s\n\n' %\
            (package, version, login, email, full_date)
        file_path = f'apps/{package}/deploy/debian/changelog'
        with open(f'{RELEASE_DIR}/{file_path}', 'r+') as f:
            content = f.read()
            f.seek(0, 0)
            f.write(record + content)
        return file_path

    @staticmethod
    def write_mpfs_version(version):
        file_path = 'lib/mpfs/__init__.py'
        with open(f'{RELEASE_DIR}/{file_path}', 'w') as f:
            f.write(f"# -*- coding: utf-8 -*-\n__version__ = '{version}.1'\n")
        return file_path

    @staticmethod
    def get_user_email(login):
        return f'{login}@yandex-team.ru'


class BuildConfig(object):
    MPFS_CONFIG_PATH = f'{MPFS_DIR}/{CONFIG_FILE}'
    RELEASE_CONFIG_PATH = f'{RELEASE_DIR}/{CONFIG_FILE}'

    def __init__(self):
        self._config = ConfigParser()
        self._config.optionxform = str
        self._config.read(self.MPFS_CONFIG_PATH)

    def get_release_number(self):
        return self._config.get('global', 'release')

    def get_version(self):
        return self._config.get('global', 'version')

    def get_packages_list(self):
        return self._config.items('packages')

    def update(self, version=None, release=None):
        with open(self.RELEASE_CONFIG_PATH, 'w') as configfile:
            if version is not None:
                self._config.set('global', 'version', str(version))
            if release is not None:
                self._config.set('global', 'release', str(release))
            self._config.write(configfile)


class SVNInfo(object):
    RELEASE_BRANCH_RELATIVE_URL = r'\^/branches/disk/mpfs_test/releases/(release-.*)'
    RELEASE_BRANCH_REGEX = re.compile(r'^release-(\d+)\.(\d+)$')

    @staticmethod
    def get_svn_info():
        info = {}
        SEPARATOR = ': '

        res = subprocess.Popen(["svn", "info"], stdout=subprocess.PIPE)
        stdout = res.communicate()[0]
        for line in stdout.splitlines():
            if SEPARATOR not in line:
                continue
            key, value = line.split(SEPARATOR, 1)
            info[key] = value

        return info

    @staticmethod
    def get_last_release():
        release_versions = []

        res = subprocess.Popen(["svn", "ls", "svn+ssh://arcadia.yandex.ru/arc/branches/disk/mpfs/releases"],
                               stdout=subprocess.PIPE)
        stdout = res.communicate()[0]
        for line in stdout.splitlines():
            if not line:
                continue
            logging.info(line)
            branch = line.decode("utf-8").strip().strip('/')
            match = SVNInfo.RELEASE_BRANCH_REGEX.search(branch)
            if match:
                release_versions.append((int(match.group(1)), int(match.group(2))))

        return max(release_versions)

    @staticmethod
    def get_relative_url():
        info = SVNInfo.get_svn_info()
        return info.get('Relative URL')

    @staticmethod
    def get_release_branch():
        relative_url = SVNInfo.get_relative_url()
        if not relative_url:
            return None

        release_branch_pattern = re.compile(SVNInfo.RELEASE_BRANCH_RELATIVE_URL)
        matched = release_branch_pattern.match(relative_url)
        if not matched:
            return

        return matched.group(1)
