import re
from startrek_client import Startrek

from disk.tasklets.startrek.proto import (
    create_st_ticket_tasklet,
    create_st_ticket_pb2
)
from tasklet.services.yav.proto import yav_pb2
from ci.tasklet.common.proto import service_pb2 as ci

PACKAGE_INFO = {
    'disk': {
        'summary': 'Протестировать mpfs для Диска, версия %(full_version)s',
        'components': {'name': 'mpfs'},
    },
    'platform': {
        'summary': 'Протестировать mpfs для Платформы, версия %(full_version)s',
        'components': {'name': 'REST API'},
    },
}

DESC_TEMPLATE = """
((%(branch_url)s Релизная ветка %(version)s))
((%(diff_url)s DIFF (trunk:%(trunk_revision)s release-%(version)s:HEAD)))
((https://teamcity.yandex-team.ru/buildConfiguration/Mpfs_Arcadia_PythonMpfsDiskApiQueue Сборка релиза на TC))
((%(ci_url)s CI процесс))
"""


class MPFSStartrekClient(Startrek):

    def __init__(self, token, useragent, queue_name, **kwargs):
        super(MPFSStartrekClient, self).__init__(useragent=useragent, token=token, **kwargs)
        self.mpfs_queue_name = queue_name
        self.mpfs_queue = self.queues[queue_name]
        self._fix_versions_map = {}

    def get_fix_version(self, name, upsert=True):
        """Получить fix version по имени и при необходимости создать"""
        if name in self._fix_versions_map:
            return self._fix_versions_map[name]

        versions = self.mpfs_queue.versions
        search_result = [v for v in versions if v.name == name]
        if len(search_result) > 1:
            raise ValueError("Fix version name should be uniq")
        elif len(search_result) == 1:
            version = search_result[0]
        elif upsert:
            version = self.versions.create(queue=self.mpfs_queue_name, name=name)
        else:
            raise KeyError("Fix version not found")
        self._fix_versions_map[version.name] = version
        return version

    def get_issues(self, query):
        """Получение тикетов"""
        query = query.encode('utf-8')
        return list(self.issues.find(query))

    def create_issue(self, **kwargs):
        return self.issues.create(**kwargs)

    def update_fix_versions(self, issue, fix_versions):
        return issue.update(fixVersions=fix_versions)


class CreateStTicketImpl(create_st_ticket_tasklet.CreateStTicketBase):

    def run(self):
        progress_msg = ci.TaskletProgress()
        progress_msg.job_instance_id.CopyFrom(self.input.context.job_instance_id)
        progress_msg.progress = 0.0
        progress_msg.module = "STARTREK"
        self.ctx.ci.UpdateProgress(progress_msg)

        st_token = self._get_st_token()
        queue_name = self.input.config.st_queue_name
        st_client = MPFSStartrekClient(token=st_token, useragent="CreateStTicket", queue_name=queue_name)
        ticket = self._create_st_ticket(st_client)

        progress_msg.progress = 1.0
        progress_msg.url = f"https://st.yandex-team.ru/{ticket.key}"
        progress_msg.status = ci.TaskletProgress.Status.SUCCESSFUL
        progress_msg.text = 'Startrek ticket created'
        self.ctx.ci.UpdateProgress(progress_msg)
        self.output.st_ticket.name = ticket.key
        self.output.st_ticket.action = create_st_ticket_pb2.StTicket.Action.CREATED

    def _get_st_token(self):
        spec = yav_pb2.YavSecretSpec(uuid=self.input.context.secret_uid, key='startrek_token')
        return self.ctx.yav.get_secret(spec).secret

    def _get_description_for_ticket(self, release):
        trunk_revision = str(self.input.context.target_revision.number)
        params = {
            'version': release.version,
            'diff_url': "https://arc.yandex-team.ru/wsvn/arc?op=comp&compare[0]=/trunk/arcadia/disk/mpfs&compare_rev[0]=%s&compare[1]=/branches/disk/mpfs/releases/release-%s&compare_rev[1]=HEAD" % (
            trunk_revision, release.version),
            'branch_url': "https://a.yandex-team.ru/arc/history/branches/disk/mpfs/releases/release-%s" % release.version,
            'trunk_revision': trunk_revision,
            'ci_url': self.input.context.ci_url,
        }
        return DESC_TEMPLATE % params

    def _create_st_ticket(self, st_client):
        config = self.input.config
        assignee = self.input.context.flow_triggered_by
        package_info = PACKAGE_INFO[config.package]
        release = Release(config.package, config.version)
        description = self._get_description_for_ticket(release)
        # создаем если нет fix version
        st_client.get_fix_version(release.fix_version, upsert=True)

        kwargs = {
            'queue': config.st_queue_name,
            'type': {'name': 'Задача'},
            'followers': [{'login': l} for l in list(config.follower_logins)],
            'assignee': {'login': assignee},
            'fixVersions': [{'name': release.fix_version}],
            'components': package_info['components'],
            'followingMaillists': list(config.following_maillists),
            'summary': package_info['summary'] % release.rvars,
            "description": description,
        }
        return st_client.create_issue(**kwargs)


class Release(object):
    version_template = "%(ver_major)s.%(ver_minor)s"
    full_version_template = "%(ver_major)s.%(ver_minor)s-%(build)s"
    package_component_map = {
        'disk': 'mpfs',
        'platform': 'REST API'
    }
    fix_version_template = "mpfs %(package)s %(ver_major)s.%(ver_minor)s"

    def __init__(self, package, version):
        if package not in self.package_component_map:
            raise ValueError('Bad package type: "%s"' % package)

        ver_major, ver_minor, build = self.parse_mpfs_version(version)

        self.package = package
        self.release_parts = {
            'package': package,
            'ver_major': ver_major,
            'ver_minor': ver_minor,
            'build': build,
        }

    @property
    def rvars(self):
        rvars = self.release_parts.copy()
        rvars['version'] = self.version
        rvars['full_version'] = self.full_version
        rvars['fix_version'] = self.fix_version
        rvars['component'] = self.component
        return rvars

    @property
    def version(self):
        return self.version_template % self.release_parts

    @property
    def fix_version(self):
        return self.fix_version_template % self.release_parts

    @property
    def component(self):
        return self.package_component_map[self.package]

    @property
    def full_version(self):
        return self.full_version_template % self.release_parts

    def __str__(self):
        return "%s %s" % (self.package, self.version)

    def __repr__(self):
        return "%s %s %s" % (self.__class__, self.package, self.full_version)

    @staticmethod
    def parse_mpfs_version(version):
        res = re.search(r'^(\d+)\.(\d+)(?:-(\d+))?$', version)
        if res:
            groups = res.groups()
            if groups[2] is None:
                groups = (groups[0], groups[1], 1)
            return [int(i) for i in groups]
        raise ValueError("Cant parse mpfs version")
