package ru.yandex.chemodan.app.tcm.actions;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.tcm.db.ConferenceMeta;
import ru.yandex.chemodan.app.tcm.services.ConferenceMetaService;
import ru.yandex.chemodan.app.tcm.zk.ShardInfo;
import ru.yandex.chemodan.web.EmptyPojo;
import ru.yandex.commune.a3.action.ActionContainer;
import ru.yandex.commune.a3.action.HttpMethod;
import ru.yandex.commune.a3.action.Path;
import ru.yandex.commune.a3.action.parameter.bind.BoundByJackson;
import ru.yandex.commune.a3.action.parameter.bind.annotation.RequestParam;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.masterSlave.WithMasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author friendlyevil
 */
@ActionContainer
public class ConferenceActions {
    private static final Logger logger = LoggerFactory.getLogger(ConferenceActions.class);
    private final ConferenceMetaService conferenceService;

    public ConferenceActions(ConferenceMetaService conferenceService) {
        this.conferenceService = conferenceService;
    }

    @Path(value = "/v1/conference", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public BalancerResponse getConferenceInfo(
            @RequestParam(value = "shortUrlId", ignoreEmpty = true, required = false) Option<String> shortUrlId,
            @RequestParam(value = "conferenceId", ignoreEmpty = true, required = false) Option<String> conferenceId,
            @RequestParam(value = "uid", ignoreEmpty = true) Option<String> uid) {
        ShardInfo shardInfo = conferenceService.getShardInfo(shortUrlId, conferenceId);
        logger.info("For getting conference with params uid={}, shortUrlId={}, conferenceId={} " +
                        "return balancerHost={} shardId={}",
                uid, shortUrlId, conferenceId, shardInfo.getBalancerHost(), shardInfo.getShardId());
        return new BalancerResponse(shardInfo.getBalancerHost());
    }

    @Path(value = "/v1/conference", methods = HttpMethod.POST)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public EmptyPojo saveConferenceInfo(@BoundByJackson ConferenceMeta conferenceMeta) {
        conferenceService.saveConferenceMeta(conferenceMeta);
        logger.info("Register conference with shortUrlId={}, conferenceId={} on shardId={}",
                conferenceMeta.getShortUrlId(), conferenceMeta.getConferenceId(), conferenceMeta.getShardId());
        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/conference", methods = HttpMethod.PUT)
    @WithMasterSlavePolicy(MasterSlavePolicy.RW_M)
    public EmptyPojo updateConferenceInfo(@BoundByJackson ConferenceMeta conferenceMeta) {
        conferenceService.updateConferenceMeta(conferenceMeta);
        logger.info("Update conference with shortUrlId={}, conferenceId={}, shardId={}, expired_at={}",
                conferenceMeta.getShortUrlId(), conferenceMeta.getConferenceId(),
                conferenceMeta.getShardId(), conferenceMeta.getExpiredAt());
        return EmptyPojo.INSTANCE;
    }

    @Path(value = "/v1/new-conference", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public BalancerResponse getNewConferenceHost(
            @RequestParam("staff") boolean isStaff,
            @RequestParam("yandexTeam") boolean isYandexTeam,
            @RequestParam(value = "uid", ignoreEmpty = true) Option<String> uid) {
        CreateConferenceSettings conferenceSettings = new CreateConferenceSettings(isStaff, isYandexTeam);
        ShardInfo shardInfo = conferenceService.getShardForNewConference(conferenceSettings);
        logger.info("For new conference with params uid={} staff={}, yandexTeam={} " +
                        "return balancerHost={} on shardId={}",
                uid, isStaff, isYandexTeam, shardInfo.getBalancerHost(), shardInfo.getShardId());
        return new BalancerResponse(shardInfo.getBalancerHost());
    }

    @Path(value = "/v1/user", methods = HttpMethod.GET)
    @WithMasterSlavePolicy(MasterSlavePolicy.R_SM)
    public BalancerResponse getHostToGetUserIngo(
            @RequestParam("staff") boolean isStaff,
            @RequestParam("yandexTeam") boolean isYandexTeam,
            @RequestParam(value = "uid", ignoreEmpty = true) Option<String> uid) {
        CreateConferenceSettings conferenceSettings = new CreateConferenceSettings(isStaff, isYandexTeam);
        ShardInfo shardInfo = conferenceService.getShardForUserInfo(conferenceSettings);
        logger.info("For getting user info with params uid={} staff={}, yandexTeam={} " +
                        "return balancerHost={} on shardId={}",
                uid, isStaff, isYandexTeam, shardInfo.getBalancerHost(), shardInfo.getShardId());
        return new BalancerResponse(shardInfo.getBalancerHost());
    }
}
