package ru.yandex.chemodan.app.tcm.db;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.spring.jdbc.JdbcTemplate3;

/**
 * @author friendlyevil
 */
public class ConferenceMetaDao {
    private final JdbcTemplate3 jdbcTemplate;

    public ConferenceMetaDao(JdbcTemplate3 jdbcTemplate) {
        this.jdbcTemplate = jdbcTemplate;
    }

    // В будующем хотим делать мультирумы для конф.
    // Все они будут в рамках одного кластера, поэтому из базки берем случайную запись
    public Option<ConferenceMeta> findByShortUrlId(String shortUrlId) {
        return jdbcTemplate.queryForOption(
                "SELECT id, short_url_id, conference_id, shard_id, expired_at, created_at from conference_meta" +
                        " WHERE short_url_id = :short_url_id order by created_at desc limit 1",
                (rs, rowNum) -> parseRow(rs),
                Cf.map("short_url_id", shortUrlId));
    }

    public Option<ConferenceMeta> findByConferenceId(String conferenceId) {
        return jdbcTemplate.queryForOption(
                "SELECT id, short_url_id, conference_id, shard_id, expired_at, created_at from conference_meta" +
                        " WHERE conference_id = :conference_id",
                (rs, rowNum) -> parseRow(rs),
                Cf.map("conference_id", conferenceId));
    }

    public ConferenceMeta create(ConferenceMeta info) {
        MapF<String, Object> params = Cf.hashMap();
        params.put( "short_url_id", info.getShortUrlId());
        params.put("conference_id", info.getConferenceId());
        params.put("shard_id", info.getShardId());
        params.put("expired_at", info.getExpiredAt().orElse((Instant) null));
        params.put("now", Instant.now());

        return jdbcTemplate.query("INSERT into conference_meta (short_url_id, conference_id, shard_id, expired_at, created_at)" +
                        " VALUES (:short_url_id, :conference_id, :shard_id, :expired_at, :now)" +
                        // фейковый update чтобы было что вернуть
                        " ON CONFLICT (conference_id) DO UPDATE SET conference_id = conference_meta.conference_id RETURNING *",
                (rs, rowNum) -> parseRow(rs), params
        ).first();
    }

    public int update(ConferenceMeta info) {
        return jdbcTemplate.update("UPDATE conference_meta set shard_id = :shard_id, expired_at = :expired_at" +
                        " WHERE short_url_id = :short_url_id and conference_id = :conference_id",
                Cf.map("short_url_id", info.getShortUrlId(),
                        "conference_id", info.getConferenceId(),
                        "shard_id", info.getShardId(),
                        "expired_at", info.getExpiredAt().orElse((Instant) null))
        );
    }

    protected ConferenceMeta parseRow(ResultSet rs) throws SQLException {
        return new ConferenceMeta(
                UUID.fromString(rs.getString("id")),
                rs.getString("short_url_id"),
                rs.getString("conference_id"),
                rs.getInt("shard_id"),
                Option.ofNullable(rs.getTimestamp("expired_at")).map(Instant::new),
                new Instant(rs.getTimestamp("created_at"))
        );
    }
}
